# -*- coding: utf-8 -*-
import re
import os
from requests import get

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'

EMAILS_GRID = 'users/hitinap/statichnyjj-email-partnjorov-v-ocv'
MANAGER_GRID = 'users/hitinap/podmena-menedzhera-magazina-v-kontaktax-robota'


def wiki_emails_substitutes(token=None):
    url = 'https://wiki-api.yandex-team.ru/_api/frontend/{}/.grid'.format(EMAILS_GRID)
    headers = {'Authorization': 'OAuth {}'.format(token),
               'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
    response = get(url, headers=headers, timeout=1200).json()

    emails = {}
    for row in response['data']['rows']:
        columns = (row[0]['raw'], row[1]['raw'])
        shops = [re.sub(r'[^0-9]', '', shop_id) for shop_id in columns[0].split(',')]
        emails.update({int(shop_id): re.sub(r'\s+', '', columns[1]) for shop_id in shops})

    return emails


def wiki_manager_substitutes(token=None):
    url = 'https://wiki-api.yandex-team.ru/_api/frontend/{}/.grid'.format(MANAGER_GRID)
    headers = {'Authorization': 'OAuth {}'.format(token),
               'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
    response = get(url, headers=headers, timeout=1200).json()

    managers = {}
    for row in response['data']['rows']:
        columns = (row[0]['raw'], row[1]['raw'])
        shops = [re.sub(r'[^0-9]', '', shop_id) for shop_id in columns[0].split(',')]
        managers.update({int(shop_id): columns[1] for shop_id in shops})

    return managers


def execute_query(query, token=None):
    from yql.api.v1.client import YqlClient
    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        column_names = result.column_names
        for record in result.get_iterator():
            yield {column_names[i]: val for i, val in enumerate(record)}


def write_table(table_write_path, table_data, token=None, proxy='hahn'):
    import yt.wrapper as yt
    client = yt.client.Yt(token=token, proxy=proxy)
    client.write_table(yt.TablePath(table_write_path, append=True),
                       table_data, raw=False)


def create_comment(shop_id, orders, email):
    text = 'Здравствуйте!\n\n'
    text += 'У магазина есть необработанные заказы, срок доставки которых уже прошел. '
    text += 'Пожалуйста, проверьте и проставьте финальный статус в личном кабинете / передайте через API:\n'
    text += ', '.join([str(order['order_id']) for order in orders]) + '\n\n'
    text += 'Отнеситесь к этому письму с особым вниманием. Мы ожидаем финальный статус в течение 2 рабочих дней. '
    text += 'В случае отсутствия статуса заказы будут отменены в соответствии с п.9.1 Требований https://yandex.ru/legal/cpa_partners_requirements/'
    text += '\n\n--\nС уважением,\nСлужба контроля качества Яндекс.Маркета'

    return {
        'email': {
            'subject': 'Яндекс.Маркет: магазин ID {} - заказы без финальных статусов будут отменены'.format(shop_id),
            'text': text,
            'info': {
                'from': 'market-dbs-ocv@yandex-team.ru',
                'to': [email]
            }
        }
    }


def create_issue(shop_id, orders, token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    shop_data = orders[0]
    summary = 'Магазин {} (ID: {}): найдены заказы без финального статуса'.format(shop_data['shop_name'], shop_id)

    description = 'Отчет АБО: https://abo.market.yandex-team.ru/shop/{}/report\n'.format(shop_id)
    description += 'ЛК партнёра: https://partner.market.yandex.ru/shop/{}/summary\n'.format(shop_data['campaign_id'])
    description += 'Менеджер магазина: {}'.format(shop_data['manager'])

    client.issues.create(
        queue='OCV',
        summary=summary,
        type={'name': 'Задача'},
        description=description,
        components=[96133],
        idMagazina=shop_id,
        customerEmail=shop_data['contacts'].email,
        comment=create_comment(shop_id, orders, shop_data['contacts'].email),
        customerOrderNumber=', '.join([str(order['order_id']) for order in orders]),
        defectOrders=len(orders),
        merchant=shop_data['shop_name'],
        campaignIDs=shop_data['campaign_id']
    )


YT_WRITE_PATH = '//home/market-quality-control/robot-johnny/without_final_status'


def run_upload(yt_token, st_token):
    with open(os.path.join(os.path.dirname(__file__), 'query.yql'), 'r') as f:
        yql_query = f.read()

    emails_substitutes = wiki_emails_substitutes(token=st_token)
    manager_substitutes = wiki_manager_substitutes(token=st_token)

    shops = {}
    for record in execute_query(yql_query, token=yt_token):
        if record['shop_id'] in emails_substitutes and record['contacts'] is not None:
            record['contacts'].email = emails_substitutes[record['shop_id']]
        if record['shop_id'] in manager_substitutes:
            record['manager'] = manager_substitutes[record['shop_id']]
        shops.setdefault(record['shop_id'], []).append(record)

    newly_orders = []
    for shop_id, orders in shops.items():
        create_issue(shop_id, orders, token=st_token)
        newly_orders.extend([{'order_id': order['order_id'], 'partner_id': shop_id} for order in orders])

    write_table(YT_WRITE_PATH, newly_orders, yt_token)
