# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from datetime import timedelta, datetime

TAGS = ['turbo_daily', 'pwa_daily']
STARTREK_URL = 'https://st-api.yandex-team.ru'


class Transmitter:
    def __init__(self):
        self.current_date = datetime.now().strftime("%Y-%m-%dT%H:%M:%S")
        self.previous_date = (datetime.today() -
                              timedelta(hours=1, minutes=10)).strftime("%Y-%m-%dT%H:%M:%S")

    def _get_previous_issues(self, token, tag):
        from startrek_client import Startrek
        client = Startrek(
            useragent='python',
            base_url=STARTREK_URL,
            token=token
        )

        component = '88127' if tag == 'turbo_daily' else '88128'
        issues = client.issues.find(
            filter={'queue': 'TURBOQUALITY', 'components': component,
                    'created': {'from': (datetime.today() -
                                         timedelta(days=30)).strftime("%Y-%m-%d"), 'to': self.current_date}},
            per_page=100
        )

        connections = {}
        for issue in issues:
            domain = issue.summary.split('(')[1].split(')')[0]
            if domain in connections.keys():
                connections[domain].append({'relationship': 'relates', 'issue': issue.key})
            else:
                connections[domain] = [{'relationship': 'relates', 'issue': issue.key}]
        existing = [issue.summary.split(']')[0].replace('[', '') for issue in issues]
        return (existing, connections)

    @staticmethod
    def _get_pwa_description(domain, user, problems, canonical):
        summary = '[%s] У партнёра PWA (%s) может быть массовая проблема' % (canonical.key, domain)

        verified_offer = None
        feed = None
        canonical_description = canonical.description.split('*')
        for string in canonical_description:
            if 'Фид на сайте' in string:
                feed = string.replace(' ', '').replace('\r', '').replace('\n', '').replace('Фиднасайте-', '')
            elif 'Название оффера' in string:
                verified_offer = string.replace('\r', '').replace('\n', '').replace(' Название оффера - ', '')\
                    .replace('!', '').replace('(зел)', '')

        description = \
'''Пользователь {user} подозревает, что у магазина может быть массовая проблема.
----
**Фид магазина:** {feed}
**Админка приложений:** https://turboapps.yandex.ru/console/applications#page-size=15&page-number=0&filter-field=title:{domain}
----
**Проблема:** {description_problems}
**Проверяемый оффер:** %%{verification_offer}%%
**Тикет проверки:** {key}
        '''.format(
            user=user,
            feed=feed,
            domain=domain,
            description_problems=', '.join(problems),
            verification_offer=verified_offer,
            key=canonical.key
        )
        return (summary, description)

    @staticmethod
    def _get_turbo_description(domain, user, problems, canonical):
        summary = '[%s] У партнёра Турбо (%s) может быть массовая проблема' % (canonical.key, domain)

        verified_offer = None
        canonical_description = canonical.description.split('*')
        for string in canonical_description:
            if 'Название оффера' in string:
                verified_offer = string.replace('\r', '').replace('\n', '').replace(' Название оффера - ', '')\
                    .replace('!', '').replace('(зел)', '')

        description = \
'''Пользователь {user} подозревает, что у магазина может быть массовая проблема.
----
**Турбо:** https://yandex.ru/turbo?text={domain}%2Fyandexturbocatalog%2F
**Таблица бана:** https://abo.market.yandex-team.ru/turbo/shop/ban
----
**Проблема:** {description_problems}
**Проверяемый оффер:** %%{verification_offer}%%
**Тикет проверки:** {key}
        '''.format(
            user=user,
            domain=domain,
            description_problems=', '.join(problems),
            verification_offer=verified_offer,
            key=canonical.key
        )
        return (summary, description)

    def run_transfer(self, token):
        from startrek_client import Startrek
        client = Startrek(
            useragent='python',
            base_url=STARTREK_URL,
            token=token
        )

        for tag in TAGS:
            origin_issues = client.issues.find(
                filter={'queue': 'ASQUESTION', 'components': '73977', 'tags': tag,
                        'resolution': 'fixed', 'resolved': {'from': self.previous_date, 'to': self.current_date}},
                per_page=100
            )
            existing, connections = self._get_previous_issues(token, tag)

            transfer_counter = 0
            for issue in origin_issues:
                problems = [item['text'] for item in issue.checklistItems if item['checked']]

                if len(problems) != 0 and issue.key not in existing:
                    domain = issue.summary.replace(' ', '')
                    user = issue.assignee.display if issue.assignee is not None else 'None'
                    links = connections[domain] if domain in connections.keys() else []

                    summary, description = self._get_pwa_description(
                        domain=domain,
                        user=user,
                        problems=problems,
                        canonical=issue
                    ) if tag == 'pwa_daily' else self._get_turbo_description(
                        domain=domain,
                        user=user,
                        problems=problems,
                        canonical=issue
                    )

                    transfer_counter = transfer_counter + 1
                    component = 'Турбо - массовые проблемы' if tag == 'turbo_daily' \
                        else 'Бесплатные фиды - массовые проблемы'
                    client.issues.create(
                        queue='TURBOQUALITY',
                        summary=summary,
                        type={'name': 'Задача'},
                        description=description,
                        links=links,
                        components=component
                    )
            logging.info('Successfully transferred %i issues on %s vmk problems.' % (transfer_counter, tag))


class VmkProblemsTransmitter(sdk2.Task):
    '''
    Description
    '''

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Running every hour every day.'
        max_restarts = 3
        dump_disk_usage = False
        fail_on_any_error = True

        oauth_token = sdk2.parameters.YavSecret(
            'OAuth ST token',
            required=True
        )

    def on_execute(self):
        token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]
        Transmitter().run_transfer(token=token)
