# -*- coding: utf-8 -*-
import logging
import pytz
import functools

from datetime import datetime
from dateutil.parser import parse as parseIso8601


def with_logging(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        from yt.wrapper.errors import YtTabletTransactionLockConflict
        from yt.wrapper.errors import YtProxyUnavailable
        from yt.wrapper.errors import YtTokenError

        try:
            func(*args, **kwargs)
        except YtTokenError:
            logging.error('Authorization failed -> invalid token')
        except YtProxyUnavailable:
            logging.error('Cluster is currently unavailable')
        except YtTabletTransactionLockConflict:
            logging.error('Can`t take exclusive lock')
        except Exception as e:
            logging.error('Unregistered exception -> %s' % e)
        else:
            logging.info('Node was successfully deleted')

    return wrapper


class Cleaner(object):
    def __init__(self, token, proxy='hahn'):
        import yt.wrapper as yt
        self.yt_client = yt.client.Yt(proxy=proxy, token=token)

    @with_logging
    def remove(self, node, recursive=False):
        self.yt_client.remove(node, force=True, recursive=recursive)

    @staticmethod
    def _is_node_expired(attributes, ttl):
        time_range = datetime.now(pytz.utc) - parseIso8601(attributes['modification_time'])
        if time_range.seconds/3600 > ttl:
            return True
        return False

    @staticmethod
    def _is_not_excluded(node, root, exclude):
        for path in exclude + [root]:
            if path.endswith('*') and node.startswith(path[:-1]):
                return False
            elif str(node) == path:
                return False
        return True

    @with_logging
    def remove_empty_nodes(self, root, exclude=[]):
        """Search and remove empty directories"""

        for node in filter(
            lambda obj: self._is_not_excluded(obj, root, exclude),
            self.yt_client.search(root, attributes=['recursive_resource_usage'], node_type=['map_node'])
        ):
            if node.attributes['recursive_resource_usage']['chunk_count'] == 0:
                self.remove(node, recursive=True)

    @with_logging
    def remove_expired(self, root, ttl=12, exclude=[]):
        """Search and remove expired nodes"""

        for node in filter(
            lambda obj: self._is_not_excluded(obj, root, exclude),
            self.yt_client.search(root, attributes=['modification_time'], node_type=['table', 'file'])
        ):
            if self._is_node_expired(node.attributes, ttl=ttl):
                self.remove(node)

    def run_cleaner(self, root, ttl=12, exclude=[]):
        """Start full cleanup process in the specified cluster directory"""

        self.remove_expired(root, ttl=ttl, exclude=exclude)
        self.remove_empty_nodes(root, exclude=exclude)
