# -*- coding: utf-8 -*-
import os
import functools
from json import loads
from multiprocessing import Pool
from datetime import datetime, timedelta

from startrek_task import StartrekTask

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def write_table(table_write_path, table_data, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)

    if table_write_path not in yt.search(table_write_path.rpartition('/')[0], node_type=['table'],
                                         client=client):
        client.create('table', table_write_path,
                      attributes={'schema': StartrekTask().schema})

    client.write_table(yt.TablePath(table_write_path, append=False), table_data,
                       raw=False)


def handle_table_data(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return {x['key']: x for x in func(*args, **kwargs)}

    return wrapper


@handle_table_data
def read_table(table_read_path, token=None, proxy='hahn'):
    import yt.wrapper as yt
    client = yt.client.Yt(token=token, proxy=proxy)
    return client.read_table(yt.TablePath(table_read_path), format='json', raw=False)


def collect_issues(filter_, token=None):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control', base_url=STARTREK_URL,
                      token=token, timeout=1600)

    return client.issues.find(filter=filter_, perPage=100)


def is_node_exists(path, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)
    return client.exists(path)


def task_configure(params, st_token=None, yt_token=None):
    time_range = (datetime.now() - timedelta(hours=8)).strftime("%Y-%m-%dT%H:%M:%S")
    task = StartrekTask()
    table = {}

    if not is_node_exists(params['path'], yt_token):
        time_range = params['from_date']
    else:
        table = read_table(params['path'], yt_token)

    params['filter'].update({'updated': {'from': time_range}})
    issues = collect_issues(params['filter'], token=st_token)

    for issue in issues:
        table[issue.key] = task.process_issue(issue, fields=params['fields'])

    write_table(params['path'], list(table.values()), token=yt_token)


def run(st_token=None, yt_token=None):
    with open(os.path.join(os.path.dirname(__file__), 'sequence.json'), 'r') as f:
        sequence = loads(f.read())

    pool = Pool(processes=2)
    pool.map(functools.partial(task_configure,
                               st_token=st_token,
                               yt_token=yt_token), sequence)
