# -*- coding: utf-8 -*-
from json import dumps
from datetime import timedelta
from dateutil.parser import parse as parseIso8601

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


class StartrekTask:

    @property
    def schema(self):
        return [
            {"name": "key", "type": "string"},
            {"name": "assignee", "type": "string"},
            {"name": "components", "type": "string"},
            {"name": "createdAt", "type": "string"},
            {"name": "createdBy", "type": "string"},
            {"name": "resolvedAt", "type": "string"},
            {"name": "resolvedBy", "type": "string"},
            {"name": "resolution", "type": "string"},
            {"name": "status", "type": "string"},
            {"name": "summary", "type": "string"},
            {"name": "sla", "type": "string"},
            {"name": "tags", "type": "string"},
            {"name": "type", "type": "string"},
            {"name": "links", "type": "string"},
            {"name": "fields", "type": "string"}
        ]

    @staticmethod
    def getLocalFieldName(field):
        return field.split('--')[1] if '--' in field else field

    @staticmethod
    def parse_sla(sla):
        sla.pop('id')
        return sla

    @staticmethod
    def getFieldValue(field):
        from startrek_client.objects import Reference
        if isinstance(field, Reference):
            field = vars(field)['_value']

        return field

    def process_issue(self, issue, fields=[]):
        return {
            'key': issue.key,
            'assignee': dumps({'login': issue.assignee.id, 'name': issue.assignee.display}, ensure_ascii=False)
            if issue.assignee is not None else None,
            'components': dumps([c.id for c in issue.components], ensure_ascii=False),
            'createdAt': (parseIso8601(issue.createdAt) + timedelta(hours=3)).strftime("%Y-%m-%dT%H:%M:%S"),
            'createdBy': dumps({'login': issue.createdBy.id, 'name': issue.createdBy.display}, ensure_ascii=False),
            'resolvedAt': (parseIso8601(issue.resolvedAt) + timedelta(hours=3)).strftime("%Y-%m-%dT%H:%M:%S")
            if issue.resolvedAt is not None else None,
            'resolvedBy': dumps({'login': issue.resolvedBy.id, 'name': issue.resolvedBy.display}, ensure_ascii=False)
            if issue.resolvedBy is not None else None,
            'resolution': issue.resolution.display if issue.resolution is not None else None,
            'status': issue.status.display,
            'summary': issue.summary.replace('"', '\"'),
            'sla': dumps([self.parse_sla(sla) for sla in issue.sla], ensure_ascii=False) if issue.sla is not None else None,
            'links': dumps([{'key': link.object.key, 'relates': link.type.id} for link in issue.links], ensure_ascii=False),
            'tags': dumps([tag for tag in issue.tags], ensure_ascii=False),
            'type': issue.type.display,
            'fields': dumps({self.getLocalFieldName(field): self.getFieldValue(getattr(issue, field))
                             for field in fields if hasattr(issue, field)}, ensure_ascii=False)
        }
