# *- coding: utf-8 -*-

import os
import logging
import time

from metro_colors_resource import RTLINE_METRO_COLORS

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.BuildRTLine import rtline_resource_types


class MasstransitSnippetsGenerating(sdk2.Task):
    """ Generating masstransit snippets via object routing """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 4 * 60 * 60  # 4h

        with sdk2.parameters.Group('Resources') as resources_block:
            masstransit_snippets_id = sdk2.parameters.String('Masstransit snippets resource id')
            dump_to_indexerproxy_id = sdk2.parameters.String('Dump to indexerproxy resource id')
            metro_colors_id = sdk2.parameters.String('Metro colors resource id')

        with sdk2.parameters.Group('GeoSaaS parameters') as saas_block:
            saas_host = sdk2.parameters.String('GeoSaas saas_host', default='saas-searchproxy-maps-prestable.yandex.net', required=True)
            saas_port = sdk2.parameters.String('GeoSaas port', default='17000', required=True)
            saas_service = sdk2.parameters.String('GeoSaas service', default='maps_graph_rus_no_traffic', required=True)
            saas_object_types = sdk2.parameters.String('Type of objects to find', default='633', required=True)

        with sdk2.parameters.Group('Generating parameters') as generating_block:
            generating_rects = sdk2.parameters.List('Specified rectangles')

        with sdk2.parameters.Group('YT Parameters') as yt_block:
            yt_token_vault_name = sdk2.parameters.String('Vault name to extract YT token', default='YT_TOKEN', required=True)
            yt_token_vault_owner = sdk2.parameters.String('Vault owner to extract YT token', default='EXTMAPS', required=True)
            yt_server = sdk2.parameters.String('YT server name', default='hahn', required=True)
            yt_jobs_count = sdk2.parameters.String('YT jobs count', default='600', required=True)
            yt_company_table = sdk2.parameters.String('Table with companies', default='//home/sprav/altay/prod/export/exported/company', required=True)
            yt_kv_table = sdk2.parameters.String('Output kv table', default='//home/extdata/masstransit-snippets/kv', required=True)
            yt_days_to_expire = sdk2.parameters.String('Tables TTL in days', default='7', required=True)

        with sdk2.parameters.Group('Dump parameters') as dump_block:
            dump_key_vault_name = sdk2.parameters.String('Vault name to extract KV key', default='GEO_SAAS_KV_KEY', required=True)
            dump_key_vault_owner = sdk2.parameters.String('Vault owner to extract KV key', required=True)
            dump_mode = sdk2.parameters.String('Mode', default='yt', required=True)
            dump_host = sdk2.parameters.String('Host', default='saas-indexerproxy-maps-prestable.yandex.net', required=True)
            dump_mr_server = sdk2.parameters.String('mr server', default='hahn.yt.yandex.net:80', required=True)
            dump_output_table = sdk2.parameters.String('mr dump output table', default='//home/extdata/masstransit-snippets/kv_dump_output', required=True)
            dump_mr_processor = sdk2.parameters.String('mr processor', default='cards_geo', required=True)
            dump_mr_action = sdk2.parameters.String('mr action', default='atModify', required=True)
            dump_mr_processor_env = sdk2.parameters.String('mr prefix', default='_masstransit/1.x', required=True)

            # redundant paremeters in stable
            dump_mr_prefix = sdk2.parameters.String('mr prefix presence, only for prestable')
            dump_deadline = sdk2.parameters.String('number of days to expire, only for prestable')

    def on_execute(self):
        os.environ['YT_TOKEN'] = self.get_yt_token()

        logging.info('Start executing')
        self.generate_key_value_table()
        self.dump_to_indexerproxy()
        logging.info('Finish executing')

    def get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)

    def get_dump_token(self):
        return sdk2.Vault.data(self.Parameters.dump_key_vault_owner, self.Parameters.dump_key_vault_name)

    def get_masstransit_snippets(self):
        masstransit_snippets_resource_id = self.Parameters.masstransit_snippets_id
        if not masstransit_snippets_resource_id:
            masstransit_snippets = sdk2.Resource.find(
                rtline_resource_types.RTLINE_MASSTRANSIT_SNIPPETS,
                attrs={"released": "stable"}).first()
            if not masstransit_snippets:
                raise errors.SandboxTaskFailureError('Can\'t get released masstransit_snippets resource')

            masstransit_snippets_resource_id = masstransit_snippets.id

        logging.info('masstransit_snippets resource id: {}'.format(masstransit_snippets_resource_id))
        return channel.task.sync_resource(masstransit_snippets_resource_id)

    def get_dump_to_indexerproxy(self):
        dump_to_indexerproxy_resource_id = self.Parameters.masstransit_snippets_id
        if not dump_to_indexerproxy_resource_id:
            dump_to_indexerproxy = sdk2.Resource.find(
                rtline_resource_types.RTLINE_DUMP_TO_INDEXERPROXY,
                attrs={"released": "stable"}).first()
            if not dump_to_indexerproxy:
                raise errors.SandboxTaskFailureError('Can\'t get released dump_to_indexerproxy resource')

            dump_to_indexerproxy_resource_id = dump_to_indexerproxy.id

        logging.info('dump_to_indexerproxy id: {}'.format(dump_to_indexerproxy_resource_id))
        return channel.task.sync_resource(dump_to_indexerproxy_resource_id)

    def get_metro_colors(self):
        metro_colors_resource_id = self.Parameters.metro_colors_id
        if not metro_colors_resource_id:
            metro_colors = sdk2.Resource.find(
                RTLINE_METRO_COLORS,
                attrs={"released": "stable"}).first()
            if not metro_colors:
                raise errors.SandboxTaskFailureError('Can\'t get released metro_colors resource')

            metro_colors_resource_id = metro_colors.id

        logging.info('metro_colors id : {}'.format(metro_colors_resource_id))
        return channel.task.sync_resource(metro_colors_resource_id)

    def generate_key_value_table(self):
        logging.info('Starting generating key-value table')

        argv = [
            self.get_masstransit_snippets(),
            '--saas-host', self.Parameters.saas_host,
            '--saas-port', self.Parameters.saas_port,
            '--saas-service', self.Parameters.saas_service,
            '--object-types', self.Parameters.saas_object_types,
            '--yt-server-name', self.Parameters.yt_server,
            '--jobs-count', self.Parameters.yt_jobs_count,
            '--company-table', self.Parameters.yt_company_table,
            '--kv-table', self.Parameters.yt_kv_table,
            '--load-metro-colors', self.get_metro_colors(),
            '--days-to-expire', self.Parameters.yt_days_to_expire
        ]

        for rect in self.Parameters.generating_rects:
            argv.append('--rect')
            argv.append(rect)

        proc_generate_kv = process.run_process(
            argv,
            log_prefix='masstransit_snippets'
        )

        return proc_generate_kv.returncode

    def dump_to_indexerproxy(self):
        """ About GeoSaas KV: https://wiki.yandex-team.ru/geosaas/cards_geo/ """

        logging.info('Starting filling values in Saas KV')

        argv = [
            self.get_dump_to_indexerproxy(),
            '--mode', self.Parameters.dump_mode,
            '-k', self.get_dump_token(),
            '-h', self.Parameters.dump_host,
            '--mr-server', self.Parameters.dump_mr_server,
            '--mr-input', self.Parameters.yt_kv_table,
            '--mr-output', self.Parameters.dump_output_table,
            '--mr-processor', self.Parameters.dump_mr_processor,
            '--action=' + self.Parameters.dump_mr_action,
            '--mr-processor-env', self.Parameters.dump_mr_processor_env,
            '--mr-jobs', self.Parameters.yt_jobs_count
        ]

        if self.Parameters.dump_mr_prefix != '':
            argv.append('--prefix')
            argv.append(self.Parameters.dump_mr_prefix)

        if self.Parameters.dump_deadline != '':
            n_days = int(self.Parameters.dump_deadline)
            unix_minutes = int(time.time()) / 60
            unix_minutes += n_days * 24 * 60
            argv.append('--deadline')
            argv.append(str(unix_minutes))

        proc_dump = process.run_process(
            argv,
            log_prefix='dump_to_indexerproxy'
        )

        return proc_dump.returncode
