# -*- coding: utf8 -*-

import logging
import os
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.svn as sdk_svn
import subprocess
import time

from sandbox.common.types.client import Tag
from sandbox.projects.common.nanny import nanny
from sandbox.projects import DeployNannyDashboard
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxSvnUrlParameter, \
    SandboxStringParameter, SandboxSelectParameter
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE, RELEASE_PRESTABLE, RELEASE_TESTING, RELEASE_UNSTABLE


class ForceReleaseParameter(SandboxBoolParameter):
    """
        Release OCSP as soon as possible
    """
    name = 'force_release'
    description = 'Release OCSP ASAP'
    default_value = False


class CertSvnUrlParameter(SandboxSvnUrlParameter):
    """
        Путь до SVN директории с шардмапами
    """
    name = 'svn_cert_url'
    description = 'SVN directory with certs (svn+ssh://url/)'
    default_value = ''


class ReleaseType(SandboxSelectParameter):
    name = 'release_type'
    description = 'Release type'
    default_value = RELEASE_STABLE
    required = True
    group = 'deployment'
    choices = (
        ('stable', RELEASE_STABLE),
        ('prestable', RELEASE_PRESTABLE),
        ('testing', RELEASE_TESTING),
        ('unstable', RELEASE_UNSTABLE),
    )


class DashboardName(SandboxStringParameter):
    name = 'dashboard_name'
    description = 'Dashboard name'
    required = True
    group = 'deployment'


class DashboardFilter(SandboxStringParameter):
    name = 'dashboard_filter'
    description = 'Dashboard filter'
    required = True
    group = 'deployment'


class DashboardRecipe(SandboxStringParameter):
    name = 'dashboard_recipe'
    description = 'Deploy recipe'
    required = True
    group = 'deployment'


class FetchOcspResponse(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    """
    Generate ocsp response files for search balancers (base class)
    """

    execution_space = 1024
    required_ram = 1024

    input_parameters = [
        ForceReleaseParameter,
        CertSvnUrlParameter,
        ReleaseType,
        DashboardName,
        DashboardFilter,
        DashboardRecipe
    ]

    # dns = ctm.DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE & Tag.IPV4

    def _release_hook(self):
        deploy_task_params = {
            'deployment_task_id': self.id,
            'deployment_release_status': self.ctx[ReleaseType.name],
            'deployment_nanny_dashboard_name': self.ctx[DashboardName.name],
            'deployment_nanny_dashboard_filter': self.ctx[DashboardFilter.name],
            'deployment_nanny_dashboard_recipe': self.ctx[DashboardRecipe.name],
            'deployment_nanny_bool_wait': False

        }

        releaser = self.create_subtask(
            task_type=DeployNannyDashboard.DeployNannyDashboard.type,
            description='Deploy OCSP',
            input_parameters=deploy_task_params
        )

        self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)

    def on_execute(self):
        with self.memoize_stage.test:
            certs_dir = sdk_svn.Arcadia.get_arcadia_src_dir(
                self.ctx[CertSvnUrlParameter.name]
            )
            current_time = int(time.time())
            os.mkdir(os.path.join(os.path.abspath(''), 'ocsp'))

            counter = 0
            for cert in os.listdir(certs_dir):
                if cert.endswith('.pem') and cert.startswith('allCAs-'):
                    ocsp_uri_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-ocsp_uri',
                            '-in', '{}/{}'.format(certs_dir, cert)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='ocsp_uri',
                    )
                    ocsp_uri = ocsp_uri_proc.stdout.read().rstrip()

                    if not ocsp_uri:
                        logging.warn('WANRNING: There is no ocsp uri for {}'.format(cert))
                        continue

                    issuer_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-issuer',
                            '-in', '{}/{}'.format(certs_dir, cert)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='issuer'
                    )
                    issuer = issuer_proc.stdout.read().rstrip()

                    if issuer == 'issuer= /C=FR/O=KEYNECTIS/CN=CLASS 2 KEYNECTIS CA':
                        CA_cert = '{}/root/CA-Certplus.pem'.format(certs_dir)
                        no_nonce = False
                        header = False

                    elif issuer == 'issuer= /C=PL/O=Unizeto Technologies S.A./OU=Certum Certification Authority/CN=Certum Level IV CA':
                        CA_cert = '{}/root/CA-Unizeto.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'
                        header = False

                    elif issuer == 'issuer= /C=RU/O=Yandex LLC/OU=Yandex Certification Authority/CN=Yandex CA':
                        CA_cert = '{}/root/CA-Yandex.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'
                        header = False

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - SHA256 - G2':
                        CA_cert = '{}/root/CA-GlobalSign.pem'.format(certs_dir)
                        no_nonce = ''
                        header = ['-header', 'HOST', 'ocsp2.globalsign.com']

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - G2':
                        CA_cert = '{}/root/CA-GlobalSignSha1.pem'.format(certs_dir)
                        no_nonce = False
                        header = ['-header', 'HOST', 'ocsp2.globalsign.com']

                    else:
                        logging.warn('WANRNING: There is no CAfile for {}'.format(cert))
                        continue

                    tmp_der_file = '{}_temporary.der'.format(cert.split('.pem')[0])

                    ocsp_generate_args = [
                        'openssl', 'ocsp',
                        '-CAfile', CA_cert,
                        '-issuer', CA_cert,
                        '-cert', '{}/{}'.format(certs_dir, cert),
                        '-url', ocsp_uri,
                        '-respout', os.path.join(os.path.abspath(''), tmp_der_file)
                    ]
                    if no_nonce:
                        ocsp_generate_args.append(no_nonce)
                    if header:
                        ocsp_generate_args.extend(header)

                    try:
                        sdk_process.run_process(
                            ocsp_generate_args,
                            work_dir=certs_dir,
                            log_prefix='generate'
                        )

                    except Exception as err:
                        logging.critical(
                            'ERROR: unexpected error due receiving response for {}: {}'.format(
                                cert, err
                            )
                        )
                        continue

                    ocsp_text_proc = sdk_process.run_process(
                        [
                            'openssl', 'ocsp', '-no_nonce',
                            '-issuer', CA_cert,
                            '-cert', '{}/{}'.format(certs_dir, cert),
                            '-respin', os.path.join(os.path.abspath(''), tmp_der_file),
                            '-CAfile', CA_cert
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='verify'
                    )

                    for line in ocsp_text_proc.stdout:
                        if 'Response verify' in line and 'OK' not in line:
                            logging.critical(
                                'ERROR: unexpected error due validating response for {}: {}'.format(
                                    cert, line
                                )
                            )
                            continue
                        if 'This Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            this_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))
                        if 'Next Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            next_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))

                    if self.ctx[ForceReleaseParameter.name] or this_update < current_time < next_update:
                        os.rename(
                            os.path.join(os.path.abspath(''), tmp_der_file),
                            os.path.join(os.path.abspath(''), 'ocsp', '{}.der'.format(cert.split('.pem')[0]))
                        )
                        counter += 1
                    else:
                        logging.critical(
                            'ERR: Status times invalid in {}. This Update: {}; Next Update: {}'.format(
                                tmp_der_file,
                                this_update,
                                next_update
                            )
                        )
                        os.remove(os.path.join(os.path.abspath(''), tmp_der_file))
                        continue

            if counter == 0:
                raise Exception('There are no valid ocsp files at all')

            sdk_process.run_process(
                [
                    'tar', '-czf',
                    os.path.join(os.path.abspath(''), 'ocsp_responses.tgz'),
                    'ocsp'
                ],
                work_dir=os.path.abspath(''),
                log_prefix='tar_files'
            )

            ocsp_responses = self.create_resource(
                arch='any',
                attributes={'ttl': 1},
                description=self.descr,
                resource_path='ocsp_responses.tgz',
                resource_type=self._get_ocsp_resource_type(),
            )
            self.mark_resource_ready(ocsp_responses)

            if not self.ctx[ForceReleaseParameter.name]:
                self.wait_time(43200)

        self._release_hook()


__Task__ = FetchOcspResponse
