# -*- coding: utf-8 -*-

import inspect
import time
import datetime
import json
import requests
import logging

import sandbox.common.types.misc as ctm
from sandbox.common.types.task import ReleaseStatus
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.sandboxapi import Sandbox

import sandbox.projects
from sandbox.projects.MediaLib import ydl, get_shardmaps_from_dashboard
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import apihelpers

from sandbox.projects.MediaLib.iss_shardtracker import iss_shardtracker_api

import sandbox.projects.MediaLib.SwitchMediaDatabase as switch_database

logger = logging.getLogger()

# how many tasks can be in PRESTABLE status
CONCURRENCY_NEWDB = 2

# Attribute name of shardmap resource TTL. Sandbox specific
_TTL_ATTRIBUTE = "ttl"

# Explicit value for ttl (once released it will turn to 'inf'), days
_SHARDMAP_TTL = 30

# Attribute name of shardmap filename.
_SHARDMAP_NAME = 'shardmap_name'

# Context field: resource id for the shardmap
_CTX_RESULT_RESOURCE_ID = "result_resource_id"

# Context field: task kill timeout (Sandbox generic)
_CTX_KILL_TIMEOUT = "kill_timeout"

# Context field: shardmap filename
_CTX_SHARDMAP_FILENAME = "shardmap_filename"


class ShardmapType(SandboxStringParameter):
    def create_choises():
        result = []
        for name, resource in inspect.getmembers(sandbox.projects.resource_types):
            if inspect.isclass(resource):
                if hasattr(resource, 'shardmap') and resource.shardmap:
                    result.append((name, name))
        return result

    name = 'shardmap_type'
    description = 'Shardmap type'
    choices = create_choises()
    required = True


class IndexState(SandboxStringParameter):
    name = 'index_state'
    description = 'Index state'
    required = True


class IndexMetaState(SandboxStringParameter):
    name = 'index_meta_state'
    description = 'Index meta state'
    required = False


class UserUploadTaskId(SandboxStringParameter):
    name = 'user_upload_task_id'
    description = 'Task ID used for uploading (only for advanced users only).'


class SwitchDashboardName(SandboxStringParameter):
    name = 'switch_dashboard_name'
    description = 'Switch Dashboard name for production db checks'
    required = False


class SwitchDashboardGroups(SandboxStringParameter):
    name = 'switch_dashboard_groups'
    description = 'Switch Dashboard Groups comma separated, used for checking current shardmap for production db checks'
    required = False


class NewDBDashboardName(SandboxStringParameter):
    name = 'newdb_dashboard_name'
    description = 'NewDB Dashboard name for production db checks'
    required = False


class NewDBDashboardRecipe(SandboxStringParameter):
    name = 'newdb_dashboard_recipe_name'
    description = 'Dashboard recipe name'
    required = True


class RollbackActivateRecipe(SandboxStringParameter):
    name = 'rollback_activate_recipe'
    description = 'Rollback activate recipe for Cancel'
    required = False


class SkipPrestableCheck(SandboxBoolParameter):
    name = 'skip_prestable_check'
    description = 'Skip prestable count check'
    default_value = False


class SkipStableCheck(SandboxBoolParameter):
    name = 'skip_stable_check'
    description = 'Skip last stable on production check'
    default_value = False


class MediaStoreShardmap(SandboxTask, nanny.ReleaseToNannyTask):
    """
    Create shardmap resource
    """

    input_parameters = (IndexState, IndexMetaState, UserUploadTaskId, SwitchDashboardName,
                        SkipPrestableCheck, SkipStableCheck, NewDBDashboardName, NewDBDashboardRecipe, RollbackActivateRecipe,
                        SwitchDashboardGroups, switch_database.SwitchDashboardRecipe)

    cores = 1
    required_ram = 2048
    execution_space = 2048

    # sandbox task for upload db
    UPLOAD_TASK = ""
    WAIT_UPLOAD_TASK = ""
    DELETE_TASK = "NANNY_DELETE_SNAPSHOT_RESOURCE"

    # YDL
    YDL_TOKEN_NAME = ''

    # Solomon
    SOLOMON_URL = 'http://api.solomon.search.yandex.net/push/json'
    SOLOMON_URL_V2 = 'https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}'

    # Monitoring settings
    monitoring_sleep = 0
    monitoring_time = 0
    monitoring_telegram_chat_id = ''
    monitoring_email_to = ''
    monitoring_vault_name = ''
    monitoring_vault_owner = ''

    @property
    def switch_type(self):
        """
        Return switch type
        E.g. for:
            images returns "images"
            video  returns "video"
        """
        raise NotImplementedError('Implement get_swtich_type()')

    @property
    def shardmap_resource(self):
        return self.ctx.get(ShardmapType.name)

    def get_switch_recipe_name(self):
        return None

    def generate_all_shardmap(self, state, meta_state, shardmap_file):
        raise NotImplementedError('Implement generate_all_shardmap()')
        # E.g.
        #
        #        state = self.ctx.get(IndexState.name)
        #        shardmap_file.write(self.shardmap_entry('imgmmeta', 0, state,
        #                                                 'ImgMmetaTier0'))
        #
        #        state = self.ctx.get(IndexState.name)
        #        count = int(self.ctx.get(PrimaryShardsCount.name))
        #        for shard in range(count):
        #            shardmap_file.write(self.shardmap_entry('imgsidx', shard,
        #                                                state, 'ImgTier0'))

    def _generate_shardmap_filename(self):
        "Generate shardmap filename using index_state"

        if _CTX_SHARDMAP_FILENAME not in self.ctx:
            dt = datetime.datetime.strptime(self.ctx[IndexState.name], "%Y%m%d-%H%M%S")
            self.ctx[_CTX_SHARDMAP_FILENAME] = dt.strftime("shardmap-%s-%Y%m%d-%H%M%S.map")
        return self.ctx[_CTX_SHARDMAP_FILENAME]

    @staticmethod
    def _get_shard_size(shard_name):
        shard_full_size, shard_inc_size = None, None
        with iss_shardtracker_api() as shardtracker:
            shard_info = shardtracker.getShardInfo(shard_name)
            if hasattr(shard_info.shardInfo, 'full'):
                shard_full_size = float(shard_info.shardInfo.full.size) / 2 ** 30
            if hasattr(shard_info.shardInfo, 'incremental'):
                shard_inc_size = float(shard_info.shardInfo.incremental.size) / 2 ** 30
        return shard_full_size, shard_inc_size

    def send_stats(self, shard_prefix, shard_num, shard_state, solomon_prj, solomon_service, oauth_token=""):
        "Send various stats about shards to solomon"
        shard_name = '{}-{}-{}'.format(shard_prefix, shard_num, shard_state)
        timestamp = int(time.mktime(datetime.datetime.strptime(shard_state, '%Y%m%d-%H%M%S').timetuple()))
        shard_full_size, shard_inc_size = self._get_shard_size(shard_name)

        stats = list()
        if shard_full_size:
            stats.append(
                {
                    'sensor': 'full_size',
                    'ts': timestamp,
                    'value': shard_full_size
                }
            )
        if shard_inc_size:
            stats.append(
                {
                    'sensor': 'incremental_size',
                    'ts': timestamp,
                    'value': shard_inc_size
                }
            )
        if len(stats) < 1:
            logging.debug('Empty stats')
            return

        # post shard size to solomon
        logging.info("Make data for solomon")
        headers = {
            'Content-type': 'application/json',
        }
        data = {
            "commonLabels": {
                "project": solomon_prj,
                "cluster": 'size',
                "service": solomon_service,
            },
            "sensors": [{
                "labels": {"sensor": key['sensor']},
                "ts": key['ts'],
                "value": key['value'],
            } for key in stats],
        }

        if oauth_token:
            url = self.SOLOMON_URL_V2.format(project=solomon_prj, cluster='size', service=solomon_service)
            headers['Authorization'] = 'OAuth {}'.format(oauth_token)
        else:
            url = self.SOLOMON_URL

        response = requests.post(url, data=json.dumps(data), headers=headers)
        response.raise_for_status()
        logging.info("Send %d to solomon" % (len(stats)))

    def shardmap_entry(self, prefix, shard, state, tier):
        template = "{prefix}-{shard:03d}-00000000-000000 {prefix}-{shard:03d}-{state} {tier}\n"
        return template.format(prefix=prefix, shard=shard,
                               state=state, tier=tier)

    def on_enqueue(self):
        "Creation of resources right after start"

        SandboxTask.on_enqueue(self)
        shardmap_filename = self._generate_shardmap_filename()
        attributes = {
            media_settings.INDEX_STATE_ATTRIBUTE_NAME: self.ctx[IndexState.name],
            _TTL_ATTRIBUTE: _SHARDMAP_TTL,
            _SHARDMAP_NAME: shardmap_filename
        }

        resource = self.create_resource(
            self.descr,
            shardmap_filename,
            self.shardmap_resource,
            arch=ctm.OSFamily.ANY,
            attributes=attributes
        )
        self.ctx[_CTX_RESULT_RESOURCE_ID] = resource.id

    def on_execute(self):
        "Download shardmap file to Sandbox"

        # Download data to temporary directory
        shardmap_filename = self._generate_shardmap_filename()
        logger.info("Going to create %s", shardmap_filename)

        sdk_paths.remove_path(shardmap_filename)

        with open(shardmap_filename, 'w') as shardmap_file:
            state = self.ctx[IndexState.name]
            if self.ctx.get(IndexMetaState.name):
                meta_state = self.ctx[IndexMetaState.name]
            else:
                meta_state = state
            self.generate_all_shardmap(state, meta_state, shardmap_file)

    def on_release(self, additional_parameters):
        shardmap_filename = self._generate_shardmap_filename()

        switch_dashboard_name = self.ctx.get(SwitchDashboardName.name, None)
        switch_dashboard_groups = self.ctx.get(SwitchDashboardGroups.name, None)
        if switch_dashboard_groups:
            switch_dashboard_groups = switch_dashboard_groups.split(',')
            logger.debug('switch_dashboard_groups: {}'.format(switch_dashboard_groups))

        newdb_dashboard_name = self.ctx.get(NewDBDashboardName.name, None)
        newdb_dashboard_recipe = self.ctx.get(NewDBDashboardRecipe.name, None)

        # PIP Logic (TESTING release)
        # ===========================
        if additional_parameters['release_status'] == ReleaseStatus.TESTING:
            try:
                ydl_token = self.get_vault_data('MEDIA_DEPLOY', self.YDL_TOKEN_NAME)
                ydl.set_shardmap_point(ydl_token, self.id, self.switch_type, "Create")
            except Exception as e:
                logger.exception(e)
                self.set_info("<strong>Can't post to YDL.</strong>", do_escape=False)

        # NewDB Logic (PRESTABEL release)
        # ===============================
        if additional_parameters['release_status'] == ReleaseStatus.PRESTABLE:
            if self.ctx.get('upload_task_id') is not None:
                raise Exception('Already released to PRESTABLE and created NewDB upload.')

            # NewDB sanity anti race condition checks
            if not self.ctx[SkipPrestableCheck.name]:
                nanny_client = nanny.NannyClient(
                    api_url='http://nanny.yandex-team.ru/',
                    oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
                )

                prestable_releases = channel.sandbox.list_releases(
                    resource_type=self.shardmap_resource,
                    release_status=ReleaseStatus.PRESTABLE,
                    limit=12,
                )
                logger.debug('prestable_releases: {}'.format(prestable_releases))

                # 1. Check for count of newdb on prod
                # -----------------------------------
                self.set_info("Checking count shardmas in PRESTABLE state.")
                if len(prestable_releases) + 1 > CONCURRENCY_NEWDB:
                    raise Exception("There are more then {} tasks in PRESTABLE state: {}".format(
                        CONCURRENCY_NEWDB,
                        prestable_releases
                    ))

                # 2. Checks that last stable release has alredy deployed
                # ------------------------------------------------------
                # This is a check for preventing race condition when we want newdb
                # but the last one that marked as STABLE hasn't switched yet.
                # We'll checks if we have exacly +1.
                if len(prestable_releases) + 1 == CONCURRENCY_NEWDB and switch_dashboard_name and not self.ctx[SkipStableCheck.name]:
                    self.set_info("Checking last STABLE switch.")

                    stable_releases = channel.sandbox.list_releases(
                        resource_type=self.shardmap_resource,
                        release_status=ReleaseStatus.STABLE,
                        limit=3,
                        order_by='-release__creation_time',
                    )
                    logger.debug('stable_releases: {}'.format(stable_releases))

                    if stable_releases:
                        list_resources = apihelpers.list_task_resources(
                            task_id=stable_releases[0].id,
                            attribute_name='shardmap_name',
                            limit=1,
                            status='READY'
                        )
                        logger.debug('list_resources: {}'.format(list_resources))

                        if not list_resources:
                            raise Exception('There is no resource with shardmap_name attribute in task: {}'.format(stable_releases[0].id))

                        current_production_shardmap_name = Sandbox().get_resource_attribute(
                            list_resources[0].id,
                            u'shardmap_name'
                        )
                        logger.debug('current_production_shardmap_name: {}'.format(current_production_shardmap_name))

                        if not current_production_shardmap_name:
                            raise Exception("Couldn't get shardmap attribute form sandbox task.")

                        statuses = (u'ACTIVE', u'ACTIVATING', u'DEACTIVATE_PENDING')
                        production_shardmaps = get_shardmaps_from_dashboard(switch_dashboard_name, nanny_client, statuses,
                                                                            absent=True, groups=switch_dashboard_groups,
                                                                            recipe_name=self.get_switch_recipe_name())
                        logger.debug('production_shardmaps: {}'.format(production_shardmaps))

                        if current_production_shardmap_name not in production_shardmaps:
                            raise Exception("Last released STABLE shardmap hasn't switched on production yet: {} {}. Current prod: {}".format(
                                stable_releases[0].id,
                                current_production_shardmap_name,
                                production_shardmaps
                            ))
                    else:
                        self.set_info("No STABLE release – nothing to check.")
                        logger.warning('There is no STABLE release for: {}'.format(str(self.shardmap_resource)))

                # 3. Check that NewDB dashboard is in ACTIVE state
                # ------------------------------------------------
                # This check is for preventing destroying on CANCEL release.
                # When we should delete all snaphots that contains CANCELLED shardmap
                if newdb_dashboard_name:
                    self.set_info("Checking for ACTIVE NewDB services")
                    services = nanny_client.get_dashboard_services(newdb_dashboard_name)
                    logger.debug("Check on update. NewDB services: {}".format(services))
                    for service in services:
                        logger.debug("Check on update. NewDB service: {}".format(service))
                        service_state = nanny_client.get_service_current_state(service)
                        logger.debug("Check on update. NewDB service_state: {}".format(service_state))
                        state = service_state[u'content'][u'summary'][u'value']
                        if state not in (u'ONLINE', ):
                            logger.error('Check on update. Servise: {} not in ONLINE state: {}'.format(service, service_state))
                            raise Exception(
                                'Can\'t release to PRESTABLE: service: {} not in ONLINE state:\n\n{}'.format(service, state)
                            )

            # run upload
            upload_task_params = {
                'shardmap_task': self.type,
                'shardmap_task_id': self.id,
                'shardmap_filename': shardmap_filename,
                'newdb_dashboard': newdb_dashboard_name,
                'newdb_dashboard_recipe_name': newdb_dashboard_recipe
            }
            self.ctx['upload_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=self.UPLOAD_TASK,
                description='Upload {} shardmap: {} task_id: {}'.format(self.switch_type, shardmap_filename, self.id),
                input_parameters=upload_task_params,
                inherit_notifications=True
            ).id
            self.set_info("Start uploading new db.")

        # Switch DB Logic (STABLE release)
        # ================================
        if additional_parameters['release_status'] == ReleaseStatus.STABLE:
            if 'upload_task_id' in self.ctx:
                upload_task_id = self.ctx['upload_task_id']
            elif self.ctx.get(UserUploadTaskId.name):
                upload_task_id = self.ctx.get(UserUploadTaskId.name)
            else:
                raise Exception("You should deploy (upload) database first!")

            release_comments = additional_parameters.get('release_comments', u'')

            wait_upload_task_params = {
                'shardmap': shardmap_filename,
                'wait_upload_task_id': upload_task_id,
                'shardmap_task_to_deploy': self.id,
                'release_comments': release_comments,
                switch_database.SwitchDashboardName.name: self.ctx[SwitchDashboardName.name],
                switch_database.SwitchDashboardRecipe.name: self.ctx[switch_database.SwitchDashboardRecipe.name],
                NewDBDashboardName.name: self.ctx[NewDBDashboardName.name]
            }
            self.ctx['switch_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=self.WAIT_UPLOAD_TASK,
                description='Waiting for uploaded {} shardmap: {} task_id: {}'.format(self.switch_type, shardmap_filename, self.id),
                input_parameters=wait_upload_task_params,
                inherit_notifications=True
            ).id
            self.set_info("Want to switch db.")

        # Delete not needed DB Logic (CANCEL release)
        # ===========================================
        if additional_parameters['release_status'] == ReleaseStatus.CANCELLED:
            if self.ctx.get('delete_task_id'):
                raise Exception("Already canceled: {}".format(self.ctx.get('delete_task_id')))

            rollback_activate_recipe = self.ctx.get(RollbackActivateRecipe.name, None)
            if not newdb_dashboard_name or not rollback_activate_recipe:
                raise Exception("For Canceletion and Delete NewDB Database from Production you MUST set NewDBDashboardName "
                                "and RollbackActivateRecipe.name!")

            monitoring_params = {
                'monitoring_sleep': self.monitoring_sleep,
                'monitoring_time': self.monitoring_time,
                'monitoring_telegram_chat_id': self.monitoring_telegram_chat_id,
                'monitoring_email_to': self.monitoring_email_to,
                'monitoring_vault_name': self.monitoring_vault_name,
                'monitoring_vault_owner': self.monitoring_vault_owner,
            }

            delete_task_params = {
                'dashboard_name': newdb_dashboard_name,
                'delete_resource_task_id': self.id,
                'rollback_activate_recipe': rollback_activate_recipe,
                'is_shardmap': True,
                'wait_check': 5 * 60,  # 5 min
            }

            delete_task_params.update(monitoring_params)

            self.ctx['delete_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=self.DELETE_TASK,
                description='Cancel {} shardmap: {} task_id: {}'.format(self.switch_type, shardmap_filename, self.id),
                input_parameters=delete_task_params,
                inherit_notifications=True
            ).id

            self.set_info("Start cancelling uploaded NewDB.")
            self.set_info("Please check status on task: {}.".format(self.ctx['delete_task_id']))

        # Ticket integration logic
        # ========================
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)
