# -*- coding: utf-8 -*-

import re
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, DictRepeater
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel

from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.common.types.task import Status
from sandbox.projects.MediaLib import ydl, monitor, get_shardmaps_from_dashboard
from sandbox.projects.common.nanny import nanny
from sandbox.projects import DeployNannyDashboard
from sandbox.projects.MediaLib.media_zk import MediaZkClient
# from sandbox.projects.common import telegram_api
from sandbox.projects.common.decorators import retries

from sandbox.projects.common import utils

logger = logging.getLogger()

RETRIES = 10
DELAY = 30


class ShardmapName(SandboxStringParameter):
    name = 'shardmap'
    description = 'Shardmap'
    required = True


class SwitchDashboardName(SandboxStringParameter):
    name = 'switch_dashboard_name'
    description = 'dashboard name for switching database'
    required = True


class SwitchDashboardRecipe(SandboxStringParameter):
    name = 'swtich_dashboard_recipe'
    description = 'dashboard recipe for switching database'
    required = True


class IsGarbageDashboardRecipe(SandboxBoolParameter):
    name = 'garbage_dashboard_parametr'
    description = 'True if this switch is with garbage tier'
    default_value = False


class DeployOldShardmap(SandboxBoolParameter):
    name = 'deploy_older'
    description = 'Deploy shardmap older than in production?'
    default_value = False


class SkipUploadedCheck(SandboxBoolParameter):
    name = 'skip_uploaded_check'
    description = 'Skip uploaded check'
    default_value = False


class ServicesFromRecipe(SandboxBoolParameter):
    name = 'services_from_recipe'
    description = 'Get list of services from recipe'
    default_value = False


class SkipThumbCheck(SandboxBoolParameter):
    name = 'skip_thumb_check'
    description = 'Skip thumb check'
    default_value = False


class ShardmapTaskToDeploy(SandboxStringParameter):
    name = 'shardmap_task_to_deploy'
    description = 'Shardmap generator task id'
    required = False


class NannyTrackerTicket(SandboxStringParameter):
    name = 'nanny_ticket'
    description = 'Nanny Tracker Ticket'
    required = True


class NewDBDashboardName(SandboxStringParameter):
    name = 'newdb_dashboard_name'
    description = 'NewDB Dashboard name for upload checks'
    required = False


class ZkFlags(DictRepeater, SandboxStringParameter):
    name = 'zk_flags'
    description = 'Zookeeper flags path and data. Must be set keys: newdb and on_production'
    required = True

    # Images:
    # ZK_FLAGS = {
    #    'newdb': '/media-services/images/flags/newdb/newdb_uploaded',
    #    'on_production': '/media-services/images/flags/newdb/on_production',
    #    'thumb': '/media-services/images/flags/production_msk_imgsthwide/on_production',
    # }
    #
    # Video:
    # ZK_FLAGS = {
    #    'newdb': '/media-services/video/flags/newdb_video_sbase/on_production',
    #    'on_production': '/media-services/video/flags/vidsbase/on_production',
    #    'thumb': '/media-services/video/flags/production_msk_vidsth/on_production'
    # }


class SwitchMediaDatabase(SandboxTask):
    """
    Переключение базы
    """

    input_parameters = [ShardmapName, SwitchDashboardName, SwitchDashboardRecipe, DeployOldShardmap, SkipUploadedCheck,
                        ShardmapTaskToDeploy, NannyTrackerTicket, ZkFlags, SkipThumbCheck, NewDBDashboardName, ServicesFromRecipe, IsGarbageDashboardRecipe]

    execution_space = 1024
    SE_TAGS = {'limit1': 1}

    # SUCCESS_TELEGRAM_CHAT_ID = -1001051169994
    YDL_TOKEN_NAME = ''  # token for ydl
    SWITCH_RELEASE_TYPE = 'stable'  # катаем только стейбл шардмапы

    ZK_FLAG_NEWDB = 'newdb'
    ZK_FLAG_ON_PRODUCTION = 'on_production'
    ZK_FLAG_GARBAGE_ON_PRODUCTION = 'garbage_on_production'
    ZK_FLAG_THUMB_ON_PRODUCTION = 'thumb'

    # Monitoring settings
    monitoring_sleep = 0
    monitoring_time = 0
    monitoring_telegram_chat_id = ''
    monitoring_email_to = ''
    monitoring_vault_name = ''
    monitoring_vault_owner = ''
    monitoring_success_alert = True

    SHARDMAP_RE = re.compile('shardmap-(\d{10})-(\d{8}-\d{6})\.map')

    @property
    def switch_type(self):
        """
        Return switch type
        E.g. for:
            images returns "images"
            video  returns "video"
        """
        raise NotImplementedError('Implement get_swtich_type()')

    def sanity_checks(self):
        """Sanity Checks for overriding"""
        pass

    def deploy_success_callback(self):
        """Mission Complete Callback for overriding"""
        pass

    @classmethod
    def _get_ts_from_shardmap(cls, shardmap):
        """Get ts from shardmap name"""
        return cls._parse_shardmap(shardmap)[0]

    @classmethod
    def _get_state_from_shardmap(cls, shardmap):
        """Get state from shardmap name"""
        return cls._parse_shardmap(shardmap)[1]

    @classmethod
    def _parse_shardmap(cls, shardmap):
        """Extract ts and state from shardmap name"""
        m = cls.SHARDMAP_RE.match(shardmap)
        if m:
            return m.groups()
        else:
            raise SandboxTaskFailureError('Got invalid shardmap: %s' % shardmap)

    def initCtx(self):
        self.ctx['kill_timeout'] = 25 * 60 * 60

    # @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    # def send_telegram_notification(self, message):
    #    """
    #    Send telegram message
    #    """
    #    telegram_bot_token = self.get_vault_data('MEDIA_DEPLOY', 'telegram_token')
    #    telegram_bot = telegram_api.TelegramBot(telegram_bot_token)
    #    telegram_bot.send_message(self.SUCCESS_TELEGRAM_CHAT_ID, message, 'HTML')

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def close_nanny_ticket(self, ticket_id):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )
        nanny_client.update_ticket_status(ticket_id, 'DEPLOY_SUCCESS', 'Db switch succeeded!')

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _check_dahsboard_uploaded(self):
        dashboard_name = self.ctx.get(NewDBDashboardName.name, None)
        if not dashboard_name:
            logger.debug("NewDB services: NewDBDashboardName is empty")
            return set()

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        uploaded_shardmaps = get_shardmaps_from_dashboard(dashboard_name, nanny_client)

        return uploaded_shardmaps

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _is_shardmap_uploaded(self):
        """Проверка что шардмап выехал в newdb конфигурации"""

        # skip
        if self.ctx[SkipUploadedCheck.name]:
            self.set_info("SkipUploadedCheck is set.")
            return True

        if self.ctx[NewDBDashboardName.name]:
            # cheks using dashboard (new):
            self.set_info("Check upload using new dashboard logic.")
            return self.ctx[ShardmapName.name] in self._check_dahsboard_uploaded()
        else:
            # cheks using ZK (old):
            self.set_info("Check upload using old ZK logic.")
            with MediaZkClient() as zk:
                db_uploaded_shardmap, _ = zk.get(self.ctx[ZkFlags.name][self.ZK_FLAG_NEWDB])
                logger.info("Uploaded shardmap: {}".format(db_uploaded_shardmap))
            return db_uploaded_shardmap == self.ctx[ShardmapName.name]

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _is_shardmap_fresh(self):
        """Проверка что шардмап для переключения свежее чем шардмап в продакшн"""
        with MediaZkClient() as zk:
            on_production, _ = zk.get(self.ctx[ZkFlags.name][self.ZK_FLAG_ON_PRODUCTION])
            on_production = on_production.split()[0]
            logger.debug("Current base production shardmap: {}".format(on_production))
        return self._get_ts_from_shardmap(self.ctx[ShardmapName.name]) >= self._get_ts_from_shardmap(on_production) or self.ctx[DeployOldShardmap.name]

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _is_thumbs_ready(self):
        """Проверка что тумбы в проде готовы"""
        # TODO (vbiriukov): update to check tickets
        with MediaZkClient() as zk:
            thumbs_uploaded, _ = zk.get(self.ctx[ZkFlags.name][self.ZK_FLAG_THUMB_ON_PRODUCTION])
            logger.debug("Current thumbs shardmap: {}".format(thumbs_uploaded))
        thumbs_shardmap = thumbs_uploaded.split()[2]
        return self._get_ts_from_shardmap(thumbs_shardmap) >= self._get_ts_from_shardmap(self.ctx[ShardmapName.name]) or self.ctx[SkipThumbCheck.name]

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _update_production_shardmap(self):
        with MediaZkClient() as zk:
            if not zk.exists(self.ctx[ZkFlags.name][self.ZK_FLAG_ON_PRODUCTION]):
                zk.create(self.ctx[ZkFlags.name][self.ZK_FLAG_ON_PRODUCTION], makepath=True)
            zk.set(self.ctx[ZkFlags.name][self.ZK_FLAG_ON_PRODUCTION], value='{}'.format(self.ctx[ShardmapName.name]))
            if self.ctx[IsGarbageDashboardRecipe.name]:
                if not zk.exists(self.ctx[ZkFlags.name][self.ZK_FLAG_GARBAGE_ON_PRODUCTION]):
                    zk.create(self.ctx[ZkFlags.name][self.ZK_FLAG_GARBAGE_ON_PRODUCTION], makepath=True)
                zk.set(self.ctx[ZkFlags.name][self.ZK_FLAG_GARBAGE_ON_PRODUCTION], value='{}'.format(self.ctx[ShardmapName.name]))

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _update_shardmap_task_desciption(self):
        shardmap_task_id = self.ctx[ShardmapTaskToDeploy.name]
        shardmap_task = channel.sandbox.get_task(shardmap_task_id)
        description = shardmap_task.description + ' [SWITCHED]'
        common.rest.Client().task[shardmap_task_id] = {"description": description}

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    @monitor
    def on_execute(self):
        # creating switch task
        if 'switch_task_id' not in self.ctx:
            if 'sanity_checks' not in self.ctx:
                self.set_info("Running pre-switch checks.")

                if not self._is_shardmap_uploaded():
                    raise Exception("Shardmap is NOT uploaded!")
                else:
                    self.set_info("Shardmap is uploaded.")

                if not self._is_shardmap_fresh():
                    raise Exception("Shardmap is OLDER than current in production!")
                else:
                    self.set_info("Shardmap is fresh.")

                if not self._is_thumbs_ready():
                    raise Exception("Thumbs are NOT ready to switch!")
                else:
                    self.set_info("Thumbs is ready to switch")

                # custom sanity_checks
                self.sanity_checks()

                self.ctx['sanity_checks'] = True
                self.set_info("Done pre-switch checks.")

            # YDL post start switching
            try:
                ydl_token = self.get_vault_data('MEDIA_DEPLOY', self.YDL_TOKEN_NAME)
                ydl.set_shardmap_point(ydl_token, self.ctx[ShardmapTaskToDeploy.name], self.switch_type, "Start switching")
            except Exception as e:
                logger.exception(e)
                self.set_info("<strong>Can't post to YDL.</strong>", do_escape=False)

            self.set_info("Starting deploy dashboard.")
            deploy_task_params = {
                'deployment_task_id': self.ctx[ShardmapTaskToDeploy.name],
                'deployment_release_status': self.SWITCH_RELEASE_TYPE,
                'deployment_nanny_dashboard_name': self.ctx[SwitchDashboardName.name],
                'deployment_nanny_dashboard_recipe': self.ctx[SwitchDashboardRecipe.name],
                'services_from_recipe': utils.get_or_default(self.ctx, ServicesFromRecipe),
                'deployment_nanny_bool_wait': True
            }
            self.ctx['switch_task_id'] = self.create_subtask(
                task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                description='Switch {} database'.format(self.switch_type),
                input_parameters=deploy_task_params,
                inherit_notifications=True,
            ).id

        # Wait and go
        switch_task = channel.sandbox.get_task(self.ctx['switch_task_id'])
        if switch_task.new_status not in list(Status.Group.FINISH + Status.Group.BREAK):
            self.set_info("Waiting deploy dashboard finishing.")
            self.wait_tasks(
                [switch_task.id, ],
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True
            )

        # Check status
        if switch_task.new_status not in (Status.SUCCESS, ):
            raise SandboxTaskFailureError('Subtask is not OK: {id}'.format(id=switch_task.id))

        self.set_info("Switch has been completed")

        # close nanny ticket
        try:
            self.close_nanny_ticket(self.ctx[NannyTrackerTicket.name])
        except Exception as e:
            logger.exception("Failed to close nanny ticket: {}".format(e))
        self.set_info("Nanny ticket closed: {}".format(self.ctx[NannyTrackerTicket.name]))

        # YDL post finish switching
        try:
            ydl_token = self.get_vault_data('MEDIA_DEPLOY', self.YDL_TOKEN_NAME)
            ydl.set_shardmap_point(ydl_token, self.ctx[ShardmapTaskToDeploy.name], self.switch_type, "Finish switching")
        except Exception as e:
            logger.exception(e)
            self.set_info("<strong>Can't post to YDL.</strong>", do_escape=False)

        # Send telegram message
        # try:
        #    self.send_telegram_notification('<strong>{1}</strong> продакшн переключен на новый шардмап: {0}.'.format(self.ctx[ShardmapName.name], self.switch_type))
        # except Exception as e:
        #    logger.exception("Failed to send message to Telegram: {}".format(e))
        # self.set_info("Telegram message has been sent")

        try:
            self.deploy_success_callback()
        except Exception as e:
            logger.exception("Exception in deploy_success_callback: {}".format(e))
        self.set_info("Deployment success has been treated properly")

        # Don't catch exception here because we want to re-run it manualy.
        self._update_production_shardmap()
        self.set_info("ZK flag has been set")

        # Update desctiption of shardmap task: adding [SWITCHED]
        try:
            self._update_shardmap_task_desciption()
        except Exception as e:
            logger.exception("Failed to update decription of shardmap task: {}".format(e))
        self.set_info("Shardmap task desctiption has been updated")
