# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter, SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel

import sandbox.common.types.task as ctt

from sandbox.projects.MediaLib import monitor
from sandbox.projects.MediaLib.media_zk import MediaZkClient
from sandbox.projects import DeployNannyDashboard
from sandbox.projects.common.decorators import retries
from sandbox.projects.common import utils

logger = logging.getLogger()

RETRIES = 10
DELAY = 30


class NewDBDashboardName(SandboxStringParameter):
    name = 'newdb_dashboard'
    description = 'NewDB Dashboard name'
    required = True


class NewDBDashboardRecipe(SandboxStringParameter):
    name = 'newdb_dashboard_recipe_name'
    description = 'Dashboard recipe name'
    required = True


class ServicesFromRecipe(SandboxBoolParameter):
    name = 'services_from_recipe'
    description = 'Get list of services from recipe'
    default_value = False


class ShardmapTaskId(SandboxStringParameter):
    name = 'shardmap_task_id'
    description = 'Shardmap generator task id'
    required = True


class ZkFlagPath(SandboxStringParameter):
    name = 'zk_flag_path'
    description = 'Zookeeper newdb on_production flag '
    required = True


class ShardmapFilename(SandboxStringParameter):
    name = 'shardmap_filename'
    description = 'Zookeeper flag data'
    required = True


class WaitDeployment(SandboxIntegerParameter):
    name = 'wait_deployment'
    description = 'Период опроса состояния автодеплоя (секунды)'
    default_value = 600  # 10 minutes


class UploadMediaDatabase(SandboxTask):
    execution_space = 128
    SE_TAGS = {'limit1': 1}

    input_parameters = [NewDBDashboardName, NewDBDashboardRecipe, ShardmapTaskId, ZkFlagPath, ShardmapFilename, ServicesFromRecipe]

    UPLOAD_RELEASE_TYPE = 'prestable'

    # Monitoring settings
    monitoring_sleep = 0
    monitoring_time = 0
    monitoring_telegram_chat_id = ''
    monitoring_email_to = ''
    monitoring_vault_name = ''
    monitoring_vault_owner = ''

    @property
    def switch_type(self):
        """
        Return switch type
        E.g. for:
            images returns "images"
            video  returns "video"
        """
        raise NotImplementedError('Implement get_swtich_type()')

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _update_newdb_shardmap(self):
        with MediaZkClient() as zk:
            if not zk.exists(self.ctx[ZkFlagPath.name]):
                zk.create(self.ctx[ZkFlagPath.name], makepath=True)
            zk.set(self.ctx[ZkFlagPath.name], value=str("{}".format(self.ctx[ShardmapFilename.name])))

    def initCtx(self):
        self.ctx['kill_timeout'] = 24 * 60 * 60  # 24 hours

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    @monitor
    def on_execute(self):
        shardmap_desc = 'Shardmap: {} task_id: {}'.format(self.ctx[ShardmapFilename.name], self.ctx[ShardmapTaskId.name])

        if 'upload_task_id' not in self.ctx:
            self.set_info("Running update dashboard")

            deploy_task_params = {
                'deployment_task_id': self.ctx[ShardmapTaskId.name],
                'deployment_release_status': self.UPLOAD_RELEASE_TYPE,
                'deployment_nanny_dashboard_name': self.ctx[NewDBDashboardName.name],
                'deployment_nanny_dashboard_recipe': self.ctx[NewDBDashboardRecipe.name],
                'services_from_recipe': utils.get_or_default(self.ctx, ServicesFromRecipe),
                'deployment_nanny_bool_wait': True,
                'wait_deployment': utils.get_or_default(self.ctx, WaitDeployment)
            }

            self.ctx['upload_task_id'] = self.create_subtask(
                task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                description='Upload {} database. {}'.format(self.switch_type, shardmap_desc),
                input_parameters=deploy_task_params,
                inherit_notifications=True,
            ).id

        upload_task = channel.sandbox.get_task(self.ctx['upload_task_id'])
        if upload_task.new_status not in list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK):
            self.set_info("Waiting for {} newdb deploy. {}".format(self.switch_type, shardmap_desc))
            self.wait_tasks(
                [upload_task.id, ],
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        # Check status
        if upload_task.new_status not in (ctt.Status.SUCCESS, ):
            raise SandboxTaskFailureError('Subtask is not OK: {id}'.format(id=upload_task.id))

        self.set_info("newdb deploy completed")

        # Don't catch exception here because we want to re-run it manualy.
        self._update_newdb_shardmap()
