# coding: utf-8

import datetime
import inspect
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxSelectParameter, SandboxIntegerParameter, SandboxStringParameter
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects import resource_types


_DAYS = 14
_DEFAULT_RELEASES_PER_DAY = 1
_LIMIT = 999
_STAT_USER = 'robot_robot-video-acc'
_STAT_PASS = 'robot-video-acc_statface'
_STAT_PATH = 'upload.stat.yandex-team.ru'


class ReleasableResourceType(SandboxStringParameter):
    def generate_choices():
        result = []
        for name, resource in inspect.getmembers(resource_types):
            if inspect.isclass(resource):
                if hasattr(resource, 'releasable') and resource.releasable:
                    result.append((name, name))
        return result
    name = 'resource_type'
    description = 'Resource type'
    choices = generate_choices()
    required = True


class ReportScale(SandboxSelectParameter):
    name = 'report_scale'
    choices = [('daily', 'daily'), ('hourly', 'hourly'), ('minutely', 'minutely')]
    default_value = 'daily'
    description = 'Report date scale'


class ReportName(SandboxStringParameter):
    name = 'report_name'
    description = 'Report Name'
    required = True


class StartDate(SandboxStringParameter):
    name = 'start_date'
    description = 'Start date (format YYYYMMDD)'
    default_value = (datetime.datetime.today() - datetime.timedelta(days=_DAYS)).strftime("%Y%m%d")


class EndDate(SandboxStringParameter):
    name = 'end_date'
    description = 'End date (format YYYYMMDD)'
    default_value = (datetime.datetime.today() - datetime.timedelta(days=1)).strftime("%Y%m%d")


class TargetReleasesNumPerDay(SandboxIntegerParameter):
    name = 'target_release_num'
    description = 'Target releases number (for 1 day)'
    default_value = _DEFAULT_RELEASES_PER_DAY


class LimitListResources(SandboxIntegerParameter):
    name = 'limit_list_resources'
    description = 'Maximum resources to take into account'
    default_value = _LIMIT


class VideoMetricsStability(SandboxTask):
    """
        Cтабильность ресурса за заданный период
    """

    environment = (
        PipEnvironment('python-statface-client', '0.14.0', use_wheel=True),
    )

    input_parameters = [
        ReleasableResourceType,
        ReportScale,
        StartDate,
        EndDate,
        TargetReleasesNumPerDay,
        LimitListResources
    ]

    def get_releases_list(self, resource_type, limit):
        return channel.sandbox.list_releases(resource_type=resource_type, limit=limit)

    class ReleaseTime:
        def __init__(self, timestamp, switch_time, obj=None):
            self.timestamp = timestamp
            self.switch_time = switch_time
            self.release_obj = obj

    def get_release_times(self, releases):
        return [
            self.ReleaseTime(
                releases[n].timestamp,
                releases[n].timestamp - releases[n + 1].timestamp,
                releases[n]
            )
            for n in range(0, len(releases) - 1)
        ]

    def get_releases_for_period(self, releases, start, end):
        return list(filter(
            lambda r: datetime.datetime.fromtimestamp(r.timestamp) >= start and datetime.datetime.fromtimestamp(r.timestamp) <= end,
            releases))

    def get_target_releases_num(self, releases_per_day, start, end):
        return (end - start).days * releases_per_day

    def get_stability(self, num_releases, target):
        return float(num_releases) / target * 100

    def _parse_date(self, date):
        return datetime.datetime.strptime(date, '%Y%m%d')

    def _scale_date(self, date, scale):
        fmt = {
            'daily': '%Y-%m-%d',
            'hourly': '%Y-%m-%d %H:00:00',
            'minutely': '%Y-%m-%d %H:%M:00',
        }.get(scale, None)
        if fmt is None:
            logging.fatal('Invalid scale value')
        return date.strftime(fmt)

    def statface_upload(self, report_name, report_scale, data):
        from statface_client import StatfaceClient

        stat_pass = self.get_vault_data(_STAT_PASS)
        stat = StatfaceClient(_STAT_USER, stat_pass, _STAT_PATH)
        report = stat.get_report(report_name)
        logging.debug('upload data: {}'. format(data))
        report.upload_data(report_scale, data)

    def on_execute(self):
        """
            Список всех релизов за заданный период
        """
        resource_name = self.ctx[ReleasableResourceType.name]
        start = self._parse_date(self.ctx[StartDate.name])
        end = self._parse_date(self.ctx[EndDate.name])
        releases = self.get_releases_for_period(
            self.get_releases_list(resource_name, self.ctx[LimitListResources.name]),
            start,
            end)
        if not releases:
            self.set_info('No releases found for resource {}'.format(resource_name))
            return
        target_releases_num = self.get_target_releases_num(self.ctx[TargetReleasesNumPerDay.name], start, end)
        stability = self.get_stability(len(releases), target_releases_num)
        self.set_info('Target releases number: {}'.format(target_releases_num), do_escape=False)
        self.set_info('Releases number: {}'.format(len(releases)), do_escape=False)
        self.set_info('Stability metric is: {0:.2f}%'.format(stability))
        self.set_info('Releases:')
        self.set_info('\n'.join(map(lambda r: str(r), releases)))
