# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.task as ctt

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask

# from sandbox.projects.common.decorators import retries
from sandbox.projects.MediaLib import monitor
from sandbox.projects.common.nanny import nanny

from sandbox.projects.MediaLib.MediaStoreShardmap import NewDBDashboardName
from sandbox.projects.MediaLib.SwitchMediaDatabase import ShardmapName, ShardmapTaskToDeploy, SwitchDashboardRecipe, SwitchDashboardName


class WaitUploadTaskId(SandboxStringParameter):
    name = 'wait_upload_task_id'
    description = 'Task Id to wait for'
    required = True


class ReleaseComments(SandboxStringParameter):
    name = 'release_comments'
    description = 'Release comments'
    default_value = u''


class WaitUploadMediaDatabase(SandboxTask):
    """Переключение большой базы Видео"""

    execution_space = 1024
    SE_TAGS = {'limit1': 1}

    input_parameters = (WaitUploadTaskId, ShardmapName, ShardmapTaskToDeploy, ReleaseComments, SwitchDashboardName, NewDBDashboardName, SwitchDashboardRecipe)

    # task to run to switch db
    SWITCH_TASK = ''

    # instructions for ticket
    SWITCH_GUIDE = ""
    DASHBOARD_LINK = ""

    # cc for ticket
    CC_USERS = [
        'vbiriukov',
    ]

    # Monitoring settings
    monitoring_sleep = 0
    monitoring_time = 0
    monitoring_telegram_chat_id = ''
    monitoring_email_to = ''
    monitoring_vault_name = ''
    monitoring_vault_owner = ''

    @property
    def switch_type(self):
        """
        Return switch type
        E.g. for:
            images returns "images"
            video  returns "video"
        """
        raise NotImplementedError('Implement get_swtich_type()')

    def initCtx(self):
        self.ctx['kill_timeout'] = 27 * 60 * 60

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    @monitor
    def on_execute(self):
        self.set_info("Waiting upload newdb.")

        wait_task = channel.sandbox.get_task(self.ctx[WaitUploadTaskId.name])

        if wait_task.new_status not in list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK):
            self.wait_tasks([wait_task.id, ],
                            list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                            wait_all=True)

        if wait_task.new_status not in (ctt.Status.SUCCESS, ):
            raise SandboxTaskFailureError("Bad wait task status: {}".format(wait_task.new_status))

        self.set_info("newdb uploaded.")

        # ticket
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )

        nanny_ticket = nanny_client.create_ticket(
            queue_id='{}'.format(self.switch_type.upper()),
            title='Переключение базы {}: {}'.format(self.switch_type, self.ctx[ShardmapName.name]),
            description='''
Инструкция:
    {}

Deployments Dashboard:
    {}'''.format(self.SWITCH_GUIDE, self.DASHBOARD_LINK),
            responsible="ulgen4a",
            urgent=True,
            copy_to=self.CC_USERS
        )
        self.ctx['nanny_ticket'] = nanny_ticket['value']['id']
        logging.debug("Created Nanny ticket: {}".format(self.ctx['nanny_ticket']))
        self.set_info("Created Nanny ticket: <a href=https://nanny.yandex-team.ru/ui/#/t/{ticket}/>{ticket}</a>".format(ticket=self.ctx['nanny_ticket']),
                      do_escape=False)

        switch_task_params = {
            'shardmap': self.ctx[ShardmapName.name],
            'shardmap_task_to_deploy': self.ctx[ShardmapTaskToDeploy.name],
            'nanny_ticket': self.ctx['nanny_ticket'],
            SwitchDashboardName.name: self.ctx[SwitchDashboardName.name],
            SwitchDashboardRecipe.name: self.ctx[SwitchDashboardRecipe.name],
            NewDBDashboardName.name: self.ctx[NewDBDashboardName.name]
        }

        switch_task = SandboxTask.create_subtask(
            self,
            task_type=self.SWITCH_TASK,
            description='Switch shardmap: {}'.format(self.ctx[ShardmapName.name]),
            input_parameters=switch_task_params,
            enqueue=False,
            inherit_notifications=True,
        )
        self.ctx['switch_task_id'] = switch_task.id

        nanny_client.update_ticket_status(self.ctx['nanny_ticket'], 'IN_QUEUE', '''Database switch sandbox task:
https://sandbox.yandex-team.ru/task/{0}/'''.format(switch_task.id))

        # add comment with Release Comments
        if self.ctx[ReleaseComments.name]:
            nanny_client.update_ticket_status(self.ctx['nanny_ticket'], 'IN_QUEUE', u'''Release Comments:
{0}'''.format(self.ctx[ReleaseComments.name]))


__Task__ = WaitUploadMediaDatabase
