# -*- coding: utf-8 -*-

import logging
import collections
from functools import wraps

from sandbox.common.types.task import Status
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter

from sandbox.projects import MediaTaskMonitoring
from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types


logger = logging.getLogger()


def add_notify(params, email_success, email_fail):
    params.update({'notify_via': 'email',
                          'notify_if_finished': email_success,
                          'notify_if_failed': email_fail, })


class MonitoringSleep(SandboxIntegerParameter):
    name = 'monitoring_sleep'
    description = 'Monitoring Sleep'
    default_value = 0


class MonitoringTime(SandboxIntegerParameter):
    name = 'monitoring_time'
    description = 'Monitoring Time'
    default_value = 0


class MonitoringTelegramChatId(SandboxStringParameter):
    name = 'monitoring_telegram_chat_id'
    description = 'Telegram Chat ID'
    default_value = ''


class MonitoringEmailTo(SandboxStringParameter):
    name = 'monitoring_email_to'
    description = 'email addres for monitoring'
    default_value = ''


class MonitoringVaultName(SandboxStringParameter):
    name = 'monitoring_vault_name'
    description = 'vault_name for telegram key'
    default_value = ''


class MonitoringVaultOwner(SandboxStringParameter):
    name = 'monitoring_vault_owner'
    description = 'vault_owner for telegram key'
    default_value = ''


monitoring_input_parameters = (MonitoringSleep, MonitoringTime, MonitoringTelegramChatId, MonitoringEmailTo,
                               MonitoringVaultName, MonitoringVaultOwner, )


def monitor(func):
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        if 'monitor_task_id' not in self.ctx:
            logger.debug('Start monitor task: {}'.format(self.id))

            if hasattr(self, 'monitoring_success_alert'):
                monitoring_success_alert = self.monitoring_success_alert
            else:
                monitoring_success_alert = self.ctx.get('monitoring_success_alert', False)

            monitor_task_params = {
                'monitoring_sleep': self.monitoring_sleep if hasattr(self, 'monitoring_sleep') else self.ctx['monitoring_sleep'],
                'monitoring_time': self.monitoring_time if hasattr(self, 'monitoring_time') else self.ctx['monitoring_time'],
                'monitoring_telegram_chat_id': self.monitoring_telegram_chat_id if hasattr(self, 'monitoring_telegram_chat_id') else self.ctx['monitoring_telegram_chat_id'],
                'monitoring_email_to': self.monitoring_email_to if hasattr(self, 'monitoring_email_to') else self.ctx['monitoring_email_to'],
                'monitoring_vault_name': self.monitoring_vault_name if hasattr(self, 'monitoring_vault_name') else self.ctx['monitoring_vault_name'],
                'monitoring_vault_owner': self.monitoring_vault_owner if hasattr(self, 'monitoring_vault_owner') else self.ctx['monitoring_vault_owner'],
                'monitoring_success_alert': monitoring_success_alert,
            }
            logger.debug(monitor_task_params)
            monitor_task = self.create_subtask(
                task_type=MediaTaskMonitoring.MediaTaskMonitoring.type,
                description='Monitor task: {}'.format(self.id),
                input_parameters=monitor_task_params,
                inherit_notifications=True
            )
            self.ctx['monitor_task_id'] = monitor_task.id
            logger.debug('Monitor task_id: {}'.format(monitor_task.id))
        else:
            # start task if it's in exception state
            logger.debug('Restart monitor task after exeption: {}'.format(self.id))
            monitor_task = channel.sandbox.get_task(self.ctx['monitor_task_id'])
            if monitor_task.new_status == Status.EXCEPTION:
                channel.sandbox.server.restart_task(monitor_task.id)
            else:
                logger.error('Error restarting monitor task after exeption monitor_task_id: {}'.format(self.ctx['monitor_task_id']))

        return func(self, *args, **kwargs)

    return wrapper


def get_shardmap_name_ra(runtime_attrs):
    # Gets shardmap-xxxx-xxx.map from snapshot
    # or returns None in case of int for example
    # or raise Excetion if type is not SANDBOX_SHARDMAP

    service_shardmap_dump = runtime_attrs[
        u'content'][u'resources'].get(u'sandbox_bsc_shard', None)
    if service_shardmap_dump is None:
        logger.debug("get_shardmap_name sandbox_bsc_shard is absent.")
        return None

    if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
        production_shardmap_task_id = service_shardmap_dump[
            u'sandbox_shardmap'][u'task_id']
        shardmap_resource_type_str = service_shardmap_dump[
            u'sandbox_shardmap'][u'resource_type']
        shardmap_resource_type = getattr(resource_types, shardmap_resource_type_str)
        production_resource_id = apihelpers.get_task_resource_id(
            production_shardmap_task_id,
            shardmap_resource_type)
        production_shardmap_name = Sandbox().get_resource_attribute(
            production_resource_id,
            u'shardmap_name')
        if not production_shardmap_name:
            raise Exception(
                "Couldn't get sandbox id with production shardmap for service")
        return production_shardmap_name
    else:
        raise Exception("get_shardmap_name has got wrong chosen_type (differ from SANDBOX_SHARDMAP): {}".format(runtime_attrs))


def get_shardmaps_from_dashboard(dashboard_name, nanny_client, statuses=None, absent=False, groups=None, recipe_name=None):
    # Gets set of shardmap names from dashboard

    if statuses is None:
        statuses = (u'ACTIVE', u'PREPARED', u'DEACTIVATE_PENDING')

    services = list()
    if recipe_name:
        recipe_content = nanny_client.get_dashboard_recipe(dashboard_name, recipe_name)
        services_raw = set()
        for task in recipe_content['content']['tasks']:
            if task['data']['name'] == 'set_snapshot_target_state':
                services_raw.add(task['data']['params']['service_id'])
        services = list(services_raw)
    else:
        services = nanny_client.get_dashboard_services_groups(dashboard_name, groups)

    services_shardmaps = collections.OrderedDict()

    logger.debug("NewDB services: {}".format(services))
    for service in services:
        shardmaps = set()
        logger.debug("NewDB service: {}".format(service))
        service_state = nanny_client.get_service_current_state(service)
        logger.debug("NewDB service_state: {}".format(service_state))

        for snapshot in service_state[u'content'][u'active_snapshots']:
            logger.debug("NewDB snapshot: {}".format(snapshot))
            if snapshot[u'state'] in statuses:
                runtime_attrs = nanny_client.get_history_runtime_attrs(snapshot[u'snapshot_id'])
                logger.debug("NewDB runtime_attrs: {}".format(runtime_attrs))

                shardmap = get_shardmap_name_ra(runtime_attrs)
                if shardmap:
                    shardmaps.add(shardmap)
                    logger.debug("NewDB add shardmap: {}".format(shardmap))
                else:
                    logger.debug("NewDB shardmap is None for service {}".format(service))

        # chech if absent shardmap in service is OK (e.g. instances of int or cbrint)
        if absent and not shardmaps:
            logger.debug("NewDB skip service: {} from add to services_shardmaps because of turn off absent setting".format(service))
        else:
            services_shardmaps[service] = shardmaps
            logger.debug("NewDB services_shardmaps[{}]: {}".format(service, shardmaps))

    # intersection sets
    uploaded_shardmaps = services_shardmaps.items()[0][1]
    logger.debug("Uploaded shardmaps: {}".format(uploaded_shardmaps))
    for service_shardmaps in services_shardmaps.itervalues():
        logger.debug("Service shardmaps: {}".format(service_shardmaps))
        uploaded_shardmaps = uploaded_shardmaps & service_shardmaps

    logger.debug("NewDB uploaded_shardmaps: {}".format(uploaded_shardmaps))

    return uploaded_shardmaps
