import re
import os
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder


class BadShardmapException(Exception):
    pass


DEFAULT_SHARDMAP_NAME_RE = 'shardmap-(?P<shardmap_ts>\d+)-(?P<shardmap_date>\d+)-(?P<shardmap_time>\d+).map'
DEFAULT_SHARD_NAME_RE = '[^\s]+\s([^\s\(]*)(\(.*\))?\s.*'


class Shardmap(object):
    def __init__(self,
                 shardmap_filename,
                 shardmap_re=DEFAULT_SHARDMAP_NAME_RE,
                 shardname_re=DEFAULT_SHARD_NAME_RE,
                 shards=None,
                 shardmap_file_path=None
                 ):

        shardmap_reg = re.compile(shardmap_re)
        m = shardmap_reg.match(shardmap_filename)
        if not m:
            raise BadShardmapException("Shardmap_filename: {}".format(shardmap_filename))

        self.shardmap_filename = shardmap_filename
        self.shardmap_ts = int(m.group('shardmap_ts'))
        self.shardmap_date = m.group('shardmap_date')
        self.shardmap_time = m.group('shardmap_time')
        self.shardmap_timestamp = "{}-{}".format(self.shardmap_date, self.shardmap_time)

        if shards:
            self.shards = shards
        else:
            self.shards = None

        if shardmap_file_path:
            # Try to get shards from shardmap file
            self.shards = []
            shard_reg = re.compile(shardname_re)
            with open(shardmap_file_path, 'r') as f:
                for line in f:
                    g = shard_reg.match(line)
                    if g:
                        self.shards.append(g.group(1))

    def __lt__(self, other):
        if self.shardmap_ts == other.shardmap_ts:
            if self.shardmap_date == other.shardmap_date:
                return self.shardmap_time < other.shardmap_time
            else:
                return self.shardmap_date < other.shardmap_date
        else:
            return self.shardmap_ts < other.shardmap_ts

    def __repr__(self):
        return self.shardmap_filename


class SvnShardmapDir(object):
    def __init__(self, arcadia_url, shardmap_pattern=DEFAULT_SHARDMAP_NAME_RE):
        self.arcadia_url = arcadia_url
        self.shardmap_re = shardmap_pattern
        self.checkout_folder = make_folder('sandboxsvn_%s' % os.path.basename(os.path.normpath(self.arcadia_url)))
        Arcadia.checkout(self.arcadia_url, path=self.checkout_folder, depth='files')

    def _find_shardmaps(self):
        shardmaps = []
        for possible_shardmap in os.listdir(self.checkout_folder):
            try:
                shardmap = Shardmap(shardmap_filename=possible_shardmap,
                                    shardmap_re=self.shardmap_re,
                                    shardmap_file_path=os.path.join(self.checkout_folder, possible_shardmap)
                                    )
                shardmaps.append(shardmap)
            except BadShardmapException:
                pass

        return shardmaps

    def get_valid_shardmaps(self):
        """
        :return: list[Shardmap]
        """
        return self._find_shardmaps()

    def get_latest_shardmap(self):
        """
        :return: Shardmap
        """
        shardmaps = self._find_shardmaps()
        if shardmaps:
            return sorted(shardmaps, reverse=True)[0]
        else:
            return None
