# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types

from sandbox.projects.common.BaseGetShardmapTask import ShardmapSvnUrlParameter
from sandbox.projects.common.DeployMediaShardmap import DeployMediaShardmap
from sandbox.projects.common.utils import check_subtasks_fails

from sandbox.projects.DeployNannyShardmap import NannyServiceNameParameter

from sandbox.projects import GetMediaShardmapFromSvnTask
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ShardmapResourceType(SandboxStringParameter):
    """
        Для какого ресурса релизить полученный шардмап
    """
    name = 'shardmap_resource_type'
    description = 'Тип ресурса в sandbox'
    default_value = 'None'


class ReleaseStatus(SandboxStringParameter):
    """
        С каким статусом релизить новый шардмап
    """
    name = 'shardmap_release_type'
    description = 'С каким статусом релизить новый шардмап'
    default_value = 'testing'


class MediaShardmapBySvn(nanny.ReleaseToNannyTask, DeployMediaShardmap):
    """
        Отслеживаем новый шардмап в svn директории и деплоим его в сервис.
        После успешного завершения даём ссылку на релиз шардмапа для выкатки
        его в продакшен.
    """

    type = 'MEDIA_SHARDMAP_BY_SVN'

    input_parameters = [
        ShardmapResourceType,
        ShardmapSvnUrlParameter,
        NannyServiceNameParameter,
        ReleaseStatus
    ]

    def _get_shardmap_resource_type(self):
        try:
            return getattr(resource_types, self.ctx[ShardmapResourceType.name])
        except AttributeError:
            raise SandboxTaskFailureError(
                "Unknown resource type: %s" %
                (self.ctx[ShardmapResourceType.name]))

    def _get_shardmap_task_type(self):
        return GetMediaShardmapFromSvnTask.GetMediaShardmapFromSvnTask.type

    def on_execute(self):
        self.ctx['shardmaps_to_release'] = []

        # get oauth token
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        # wait subtasks if exists
        subtasks = channel.sandbox.list_tasks(parent_id=self.id)
        tasks_to_wait = filter(lambda x: not x.is_done(), subtasks)
        if tasks_to_wait:
            check_subtasks_fails(fail_on_first_failure=True)

        child_tasks = []
        for service_name in self.ctx[NannyServiceNameParameter.name].split(','):
            current_state = nanny_client.get_service_current_state(service_name)
            if current_state[u'content'][u'summary'][u'value'] == u'ONLINE':
                # find new shardmaps to deploy
                deployment_shardmap = self._get_shardmap_to_deploy(nanny_client, current_state)
                if deployment_shardmap is not None and deployment_shardmap not in self.ctx['shardmaps_to_release']:
                    # run deployment
                    shardmap_resource = channel.sandbox.get_resource(deployment_shardmap)
                    self.create_release(shardmap_resource.task_id, status=self.ctx['shardmap_release_type'])
                    self.ctx['shardmaps_to_release'].append(deployment_shardmap)
        if child_tasks:
            self.wait_all_tasks_stop_executing(child_tasks)


__Task__ = MediaShardmapBySvn
