# -*- coding: utf-8 -*-

import logging
import datetime as dt

from sandbox import common

from sandbox.common.types.task import Status
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter, SandboxBoolParameter
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common import utils
from sandbox.projects.common.decorators import retries


logger = logging.getLogger()

RETRIES = 10
DELAY = 5

SB_TASK_URL = 'https://sandbox.yandex-team.ru/task/{}/view'


class MonitoringSleep(SandboxIntegerParameter):
    name = 'monitoring_sleep'
    description = 'Sleep for seconds'
    default_value = 10 * 60  # 10 min


class MonitoringTaskId(SandboxStringParameter):
    name = 'monitoring_task_id'
    description = 'Task Id for monitoring'


class MonitoringTime(SandboxStringParameter):
    name = 'monitoring_time'
    description = 'Time to alert in seconds'
    default_value = 7 * 60 * 60  # 7 hours


class TelegramChatId(SandboxStringParameter):
    name = 'monitoring_telegram_chat_id'
    description = 'Chat id for telegram'
    default_value = "-1001051169994"


class Email(SandboxStringParameter):
    name = 'monitoring_email_to'
    description = 'Send emails'


class VaultName(SandboxStringParameter):
    name = 'monitoring_vault_name'
    description = 'Sandbox vault telegram token'
    default_value = 'telegram_token'


class VaultOwner(SandboxStringParameter):
    name = 'monitoring_vault_owner'
    description = 'Owner sandbox vault for telegram token'
    default_value = 'IMAGES-BASE-DEPLOY'


class SuccessAlert(SandboxBoolParameter):
    name = 'monitoring_success_alert'
    description = 'Send message after successful accomplishment'
    default_value = False


class MediaTaskMonitoring(SandboxTask):
    """Monitoring parent task"""

    type = 'MEDIA_TASK_MONITORING'

    execution_space = 1024
    cores = 1
    required_ram = 2048
    input_parameters = (MonitoringTaskId, MonitoringSleep, MonitoringTime,
                        TelegramChatId, VaultName, VaultOwner, Email, SuccessAlert)

    def initCtx(self):
        self.ctx['kill_timeout'] = 27 * 60 * 60  # 27 hours

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def notify(self, task, time=None, status="ERROR"):
        """Send notifications."""

        data = {
            'notify_status': status.upper(),
            'status': task.new_status,
            'time': time,
            'task_id': task.id,
            'task_desc': task.description,
            'task_type': task.type,
            'sb_url': SB_TASK_URL.format(task.id),
        }

        message_common = '<strong>[{notify_status}]</strong> {task_type} ({task_desc}) <a href="{sb_url}">{task_id}</a>'.format(
            **data)

        if time is not None:
            # timeouted
            message = message_common + ' running too long: {time}'.format(**data)
        else:
            # get status
            message = message_common + ' got status: {status}'.format(**data)

        logger.debug('Send message: {}'.format(message))

        # telegram
        chat_id = utils.get_or_default(self.ctx, TelegramChatId)
        if chat_id:
            vault_key = utils.get_or_default(self.ctx, VaultName)
            vault_owner = utils.get_or_default(self.ctx, VaultOwner)
            telegram_bot_token = self.get_vault_data(vault_owner, vault_key)
            telegram_bot = common.telegram.TelegramBot(telegram_bot_token)
            telegram_bot.send_message(chat_id, message, common.telegram.ParseMode.HTML)
            logger.debug('telegram message has been sent')

        # email
        email_to = self.ctx.get(Email.name, None)
        if email_to:
            channel.sandbox.send_email(email_to, None, 'Media Monitor Task {}:{} Error'.format(task.type, task.id),
                                       message, 'text/html', 'utf-8')
            logger.debug('email has been sent')

    def on_execute(self):
        parent_id = self.parent_id

        if self.ctx[MonitoringTaskId.name]:
            monitor_id = self.ctx[MonitoringTaskId.name]
        elif parent_id:
            monitor_id = parent_id
        else:
            raise SandboxTaskFailureError("No parent and not set MonitoringTaskId.")

        monitor_task = channel.sandbox.get_task(monitor_id)
        monitor_time = int(utils.get_or_default(self.ctx, MonitoringTime))

        logger.debug('Monitor id: {}'.format(monitor_id))
        logger.debug('Monitor time: {}'.format(monitor_time))
        logger.debug('Monitor task new_status: {}'.format(monitor_task.new_status))

        if monitor_task.new_status in list(self.Status.Group.FINISH + self.Status.Group.BREAK):
            if monitor_task.new_status in (Status.SUCCESS, ):
                # exit
                if utils.get_or_default(self.ctx, SuccessAlert):
                    logger.debug("send successful message")
                    self.notify(monitor_task, time=None, status="SUCCESS")
                self.ctx['finished'] = True
            elif monitor_task.new_status in (Status.RELEASING,):
                # nothing to do here
                pass
            elif monitor_task.new_status == Status.EXCEPTION:
                # exception - we can restart
                logger.debug("monitor_task.new_status == Status.EXCEPTION. task_id: {}".format(monitor_id))
                self.notify(monitor_task)
                raise Exception("Got EXCEPTION from monitored task_id: {}".format(monitor_id))
            else:
                # not restartable errors
                logger.debug("monitor_task.new_status: {}. task_id: {}".format(monitor_task.new_status, monitor_id))
                self.ctx['exit_with_error'] = True
                self.notify(monitor_task)

        elif monitor_time > 0:
            # monitor execution time
            # old start time (N.B. can produce false alarms because of DRAFT status)
            start = dt.datetime.utcfromtimestamp(monitor_task.timestamp)
            logger.debug('Old start: {}'.format(start))
            # real first ENQUEUING time
            history = channel.rest.server.task[monitor_task.id].audit[:]
            logger.debug('history: {}'.format(history))
            real_starts = [s["time"] for s in history if s["status"] == Status.ENQUEUING]
            logger.debug("real_starts: {}".format(real_starts))
            if real_starts:
                real_start = common.api.DateTime().decode(real_starts[0])
                now = dt.datetime.utcnow()
                logger.debug("Waiting taks. Real start: {} Now: {}".format(real_start, now))

                time_delta = now - real_start
                sec_delta = int((now - real_start).total_seconds())
                logger.debug('time_delta: {}'.format(time_delta))
                logger.debug('sec_delta: {}'.format(sec_delta))
                logger.debug('monitor_time: {}'.format(monitor_time))
                if sec_delta > monitor_time:
                    logger.debug("sec_delta > monitor_time, {} > {}".format(sec_delta, monitor_time))
                    self.notify(monitor_task, str(time_delta))
                    self.ctx['exit_with_timeout'] = True

        # exit with error
        if 'exit_with_error' in self.ctx:
            logger.debug("exit_with_error")
            raise SandboxTaskFailureError("Stop monitoring: monitoring task got an error.")

        # exit with timeout
        if 'exit_with_timeout' in self.ctx:
            logger.debug("exit_with_timeout")
            raise SandboxTaskFailureError("Stop monitoring: monitoring task timeouted.")

        # sleep
        if 'finished' not in self.ctx:
            sleep_time = utils.get_or_default(self.ctx, MonitoringSleep)
            logger.debug("sleep for {}".format(sleep_time))
            if sleep_time <= 0:
                raise SandboxTaskFailureError("Stop monitoring: MonitoringSleep <= 0")
            self.wait_time(sleep_time)


def create_monitoring_params(email, telegram_chat_id, switch_timeout, group_name='Monitoring'):

    class _TelegramChatId(TelegramChatId):
        default_value = telegram_chat_id
        group = group_name

    class _Email(Email):
        default_value = email
        group = group_name

    class _MonitoringTime(MonitoringTime):
        default_value = switch_timeout
        group = group_name

    return (_TelegramChatId, _Email, _MonitoringTime)


__Task__ = MediaTaskMonitoring
