import os

import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.projects import resource_types
from sandbox.projects.common.userdata import userdata_base_task, util


class MiscdataPackage(sdk_parameters.LastReleasedResource):
    name = 'miscdata_package_resid'
    description = 'Resource with miscdata package'
    resource_type = resource_types.USERFEAT_MISCDATA_PACKAGE
    group = userdata_base_task.PACKAGES_GROUP_NAME


class MiscdataPrepUpdateRun(userdata_base_task.Task):
    """
        Runs various modes of md_prep on a sample of input data. Results to be used by userdata_prep tests
    """

    type = 'MISCDATA_PREP_UPDATE'

    execution_space = 35000
    node_chunk_store_quota = 15 << 30
    forbid_chunk_storage_in_tmpfs = True
    yt_testable = True
    need_rem = False

    input_parameters = util.smart_join_params(
        userdata_base_task.Task.input_parameters,
        MiscdataPackage,
    )

    @util.lazy_property
    def dates_context(self):
        out = self.get_state_attrs()
        last_date = util.str2date(out['last_date'])
        out['base_timestamp'] = util.date2timestamp(last_date)
        return out

    def patch_config(self):
        data = {
            "beast_cur_date": self.dates_context["last_date"]
        }
        self.config_patcher.patch(
            os.path.join(self.ctx['berkanavt'], 'miscdata/scripts/miscdata/miscdata_config.py'),
            data
        )

    def get_project_bin_dir(self):
        return os.path.join(self.ctx["berkanavt"], "miscdata/bin")

    def process_mr_data(self):
        ctx = self.ctx.copy()
        ctx.update(self.dates_context)
        self.patch_config()

        ctx['tables_prefix'] = self.get_tables_prefix()
        env = (
            "MR_TABLE_PREFIX={tables_prefix} MR_OPT=stderrlevel=4,failonemptysrctable=0 "
            "MR_USER=userfeat " + self.get_client_environ_str()
        )

        pr = util.ProcessRunner()

        scripts_dir = os.path.join(self.ctx['berkanavt'], 'miscdata/scripts/miscdata')

        for py in os.listdir(scripts_dir):
            if "config" in py or not py.endswith(".py"):
                continue
            pr.add(
                self.specific_logs_prefix(py),
                env + " /skynet/python/bin/python {scripts_dir}/{script} ",
                scripts_dir=scripts_dir, script=py, **ctx
            )
        pr.run()

    def updated_result_attrs(self, attrs):
        ctx = self.dates_context
        attrs['base_timestamp'] = ctx['base_timestamp']
        return attrs


__Task__ = MiscdataPrepUpdateRun
