# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects import resource_types
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import settings as ms
from sandbox.projects.common.dolbilka import DolbilkaExecutor
from sandbox.projects.common import apihelpers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils


class MaxFailPercentage(sp.SandboxFloatParameter):
    name = 'max_fail_percentage'
    description = 'Max fail rate, in percents'
    default_value = 1.0


class MmetaFrequentStability(SandboxTask):
    """
        For https://st.yandex-team.ru/SEARCH-1205

        Starts 1 mmeta instance with production config (i.e. send subsource requests to production int instances).
        Task shoots 1 most-frequent request (vk) with huge rps (1000)
        Test fails if there are too much slow responses (more than MaxFailPercentage, 1% by default).
    """

    type = 'MMETA_FREQUENT_STABILITY'

    input_parameters = (
        sc.create_middlesearch_params(with_config=False).params +
        DolbilkaExecutor.input_task_parameters + (
            MaxFailPercentage,
        )
    )

    required_ram = 96 * 1024  # 96 Gb

    @property
    def footer(self):
        return [{
            'helperName': '',
            'content': '<h3>Amount of responses slower than 500ms = {}%</h3>'.format(
                self.ctx.get("percent_of_slow_responses", "???")
            )
        }]

    def on_execute(self):
        self._get_current_prod_config_id()
        middlesearch = sc.get_middlesearch(
            patch_queues_size=False,  # https://st.yandex-team.ru/SEARCH-1205#1448917277000
            reset_memory_limit=False
        )
        d_executor = DolbilkaExecutor()
        results = d_executor.run_sessions(self._get_plan(), middlesearch, run_once=True)

        self.ctx['results'] = results
        self._count_stats()

    def _get_plan(self):
        plan_id = utils.get_and_check_last_resource_with_attribute(
            resource_types.BASESEARCH_PLAN,
            attr_name="vk_plan",
            attr_value="yes"
        ).id
        return self.sync_resource(plan_id)

    def _count_stats(self):
        resource_stat = apihelpers.list_task_resources(
            self.id,
            resource_type=resource_types.EXECUTOR_STAT,
            arch=self.arch
        )[0]
        max_fail_percentage = float(utils.get_or_default(self.ctx, MaxFailPercentage))
        stat_lines = fu.read_lines(resource_stat.path)
        for line in reversed(stat_lines):
            # try to find amount of queries, slower than 500ms
            if "500000" in line:
                logging.info("line with info: %s", line)
                per_cent = float(line.split()[-1].strip("%()"))
                self.ctx["percent_of_slow_responses"] = per_cent
                if per_cent > max_fail_percentage:
                    eh.check_failed("Too much slow queries: {}%".format(per_cent))
                return
            if "F range" in line:
                break

        eh.fail(
            "Something wrong with stats format. "
            "Make sure that you specified bounds correctly."
        )

    def _get_current_prod_config_id(self):
        if self.ctx.get(sc.DefaultMiddlesearchParams.Config.name):
            return

        cfg_id = utils.get_and_check_last_resource_with_attribute(
            resource_type=resource_types.MIDDLESEARCH_CONFIG,
            attr_name=ms.WebSettings.testenv_middle_cfg_attr_name("jupiter_mmeta"),
        ).id
        self.ctx[sc.DefaultMiddlesearchParams.Config.name] = cfg_id


__Task__ = MmetaFrequentStability
