# -*- coding: utf-8 -*-

import time
from datetime import timedelta, datetime
from logging import error, info
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.task as sdk_task
from sandbox.projects.common.yabs.graphite import Graphite, five_min_metric, YABS_SERVERS
from sandbox.sandboxsdk.channel import channel


class BsGraphiteHosts(sdk_parameters.SandboxStringParameter):
    name = "graphite_hosts"
    description = "Comma separated list of all bsgraphite hosts"
    default_value = ",".join(YABS_SERVERS)


class ResourceType(sdk_parameters.SandboxStringParameter):
    name = "resource_type"
    description = "Type of resource for monitoring"
    default_value = "LINEAR_MODEL_BINARY_DUMP"


class Attrs(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
    name = "attrs"
    description = "attributes of resource to monitor"


class MetricPrefix(sdk_parameters.SandboxStringParameter):
    name = "metric_prefix"
    description = "metric prefix for graphite"


class FieldWithDate(sdk_parameters.SandboxStringParameter):
    name = "field_with_date"
    description = "field with update date"
    default_value = "last_log_date"


class DateFormat(sdk_parameters.SandboxStringParameter):
    name = "field_with_date_format"
    description = "field with date format"
    default_value = "%Y%m%d%H%M"


def get_latest_resource(*args, **kwargs):
    resources = channel.sandbox.list_resources(order_by="-id", limit=1, status="READY", *args, **kwargs)
    if resources:
        return resources[0]
    error("Can't find latest resource: %s", kwargs)


class Task(sdk_task.SandboxTask):
    cores = 1
    required_ram = 2 * 1024
    execution_space = 1024
    type = "MODEL_DATE_MONITOR"

    input_parameters = [
        BsGraphiteHosts,
        Attrs,
        ResourceType,
        MetricPrefix,
        FieldWithDate,
        DateFormat
    ]

    def on_execute(self):
        graphiteHosts = self.ctx[BsGraphiteHosts.name].split(",")
        graphiteObj = Graphite(graphiteHosts, timedelta(seconds=5).total_seconds())

        resource = get_latest_resource(
            resource_type=self.ctx[ResourceType.name],
            all_attrs=self.ctx[Attrs.name]
        )
        info(resource)
        date_str = resource.attributes.get(self.ctx[FieldWithDate.name])
        datetime_format = self.ctx[DateFormat.name]
        delta = datetime.now() - datetime.strptime(date_str, datetime_format)

        info("Now time is {dt}".format(dt=datetime.now()))
        info("DateStr from dump {date_str}".format(date_str=date_str))
        info("Delta from dump {delta}".format(delta=delta))
        metric_prefix = five_min_metric(self.ctx[MetricPrefix.name])
        info(metric_prefix)
        value = delta.total_seconds()

        graphiteObj.send([(metric_prefix, value, time.time())])


__Task__ = Task
