import os

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import (
    SandboxStringParameter,
    LastReleasedResource,
    SandboxBoolParameter,
    DictRepeater
)

import shutil

from sandbox.sandboxsdk.paths import get_unique_file_name, make_folder, remove_path, get_logs_folder

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers


class SrcResource(LastReleasedResource):
    """
       Resource with original files
    """
    name = 'src_resource'
    description = 'Original resource to modify'
    required = False


class AuxResources(LastReleasedResource):
    """
       Resources with auxiliary files
    """
    name = 'aux_resources'
    multiple = True
    description = 'Auxiliary resources'
    required = False


class CmdlineParameter(SandboxStringParameter):
    """
        Command for modifying original resource
    """
    name = 'cmdline'
    description = 'Shell command, placeholders: {src}, {dst}, {tmp}, {aux0}, ...'


class ResourceTypeParameter(SandboxStringParameter):
    """
        Type of resulting resource
    """
    name = "dst_resource_type"
    description = "Resulting resource type"
    choices = sorted([(rt.name, rt.name) for rt in resource_types.AbstractResource])


class CmdlineKindSelector(SandboxStringParameter):
    """
       Modifying mode
    """
    name = 'cmdline_kind'
    description = 'Action'
    choices = [
        ('overwrite', 'overwrite'),
        ('clone', 'clone'),
        ('custom', 'custom')
    ]
    sub_fields = {
        'overwrite': [ResourceTypeParameter.name, AuxResources.name],
        'clone': [],
        'custom': [ResourceTypeParameter.name, AuxResources.name, CmdlineParameter.name]
    }

    default_value = 'overwrite'


class CopySrcAttrsParameter(SandboxBoolParameter):
    """
        Set whether it is required to copy attributes of src resource
    """
    name = "copy_src_attrs"
    description = "Copy attributes of src resource"
    default_value = False


class SrcResourceType(SandboxStringParameter):
    name = "src_resource_type"
    description = "Type of resource (if concrete resource is not specified"
    choices = sorted([(rt.name, rt.name) for rt in resource_types.AbstractResource])


class SetAttrs(DictRepeater, SandboxStringParameter):
    name = 'set_attrs'
    description = "Add attributes to resource (will be formatted via task context):"
    required = False


class ModifyResource(SandboxTask):
    """
        Modifies files of original resource using specified patch command and other resource(s)
    """
    type = "MODIFY_RESOURCE"

    execution_space = 120000

    input_parameters = [
        SrcResource,
        SrcResourceType,
        AuxResources,
        CmdlineKindSelector,
        CmdlineParameter,
        ResourceTypeParameter,
        CopySrcAttrsParameter,
        SetAttrs
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        if self.ctx.get(SrcResource.name):
            src_resource = channel.sandbox.get_resource(self.ctx[SrcResource.name])
        else:
            src_resource = apihelpers.get_last_resource(resource_type=self.ctx[SrcResourceType.name])
            self.ctx[SrcResource.name] = src_resource.id
        dst_resource_type = self.ctx.get(ResourceTypeParameter.name)
        copy_src_attrs = self.ctx.get(CopySrcAttrsParameter.name)
        dst_resource = self.create_resource(
            description='Modified resource "{}"'.format(
                src_resource.description
            ),
            resource_path='',
            resource_type=dst_resource_type or src_resource.type,
            attributes=src_resource.attributes if copy_src_attrs else {},
            arch='any',
        )
        self.execution_space = (src_resource.size / 1024) * 2

        self.ctx['dst_resource_id'] = dst_resource.id

    def on_execute(self):
        cmdline_kind = self.ctx[CmdlineKindSelector.name]
        src_resource = channel.sandbox.get_resource(self.ctx[SrcResource.name])
        src_resource_path = self.sync_resource(src_resource.id)
        aux_resources = [self.sync_resource(id) for id in (self.ctx.get(AuxResources.name) or [])]

        dst_name = src_resource.file_name
        if cmdline_kind == "clone":
            if os.path.isdir(src_resource_path):
                shutil.copytree(src_resource_path, dst_name)
            else:
                shutil.copy(src_resource_path, dst_name)
        else:
            if cmdline_kind == "overwrite":
                cmdline = "cp -rf {src}/* {dst}"
                for index in xrange(len(aux_resources)):
                    cmdline += " && cp -rf {aux%d}/* {dst}" % index
            else:
                cmdline = self.ctx.get(CmdlineParameter.name)
            tmp_folder = None
            if "{tmp}" in cmdline:
                tmp_folder = make_folder(get_unique_file_name(self.abs_path(''), "cmd_tmp"))

            dst_name = src_resource.file_name

            fmt = {"aux{}".format(index): path for index, path in enumerate(aux_resources)}
            fmt["src"] = src_resource_path
            fmt["dst"] = dst_name
            fmt["tmp"] = tmp_folder

            formatted_cmdline = cmdline.format(**fmt)
            run_process([formatted_cmdline], shell=True, log_prefix=os.path.join(get_logs_folder(), "action_log"))
            if tmp_folder:
                remove_path(tmp_folder)

        self.change_resource_basename(self.ctx["dst_resource_id"], dst_name)

        if self.ctx.get(SetAttrs.name):
            for k, v in self.ctx[SetAttrs.name].items():
                k = k.format(**self.ctx)
                if isinstance(v, (str, unicode)):
                    v = v.format(**self.ctx)
                channel.sandbox.set_resource_attribute(self.ctx["dst_resource_id"], k, v)


__Task__ = ModifyResource
