# coding: utf-8

import logging
import json
import requests

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox.projects.Molly.MollyReqMiner import MollyReqMiner, MollyReqs
from sandbox.projects.MollyRun import MollyRun

log = logging.getLogger(__name__)


class MollyAggregateScanner(sdk2.Task):
    MOLLY_BASE = "https://molly.yandex-team.ru/api"
    FMT_JSON2 = 4
    ssl_verify = False  # Skynet, we look at you

    def on_execute(self):
        log.info("Entering task")

        payload = {'format': 'json'}
        url = self.MOLLY_BASE + '/v1.1/environments/'
        headers = {'Authorization': 'OAuth {}'.format(sdk2.Vault.data("MOLLY_SANDBOX_TOKEN"))}

        molly_environments = []
        try:
            r = requests.get(url, params=payload, headers=headers, verify=self.ssl_verify)
            molly_environments = r.json()
        except Exception as e:
            logging.exception(e)
            raise TaskFailure("Molly env list fetch failed")

        log.debug(json.dumps(molly_environments))

        with self.memoize_stage.first_step:
            children = []
            for item in molly_environments.get('env_list', []):
                env_id = item.get('id')
                logging.info('Creating miner subtask for {}'.format(env_id))

                subtask = MollyReqMiner(
                    self,
                    description="Running log processor for {} (subtask of {})".format(env_id, self.id),
                    owner=self.owner
                )
                subtask.max_restarts = 2
                subtask.Parameters.target_id = env_id

                logging.info("Saving subtask")
                subtask.save()
                children.append(subtask)

            logging.info("Waiting for REQ_MINER subtasks to finish")
            raise sdk2.WaitTask([c.enqueue() for c in children], list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

        with self.memoize_stage.second_step:
            children = []
            for item in molly_environments.get('env_list', []):
                env_id = item.get('id')
                logging.info('Creating runner subtask for {}'.format(env_id))

                binary_resource = sdk2.Resource.find(
                    resource_type=MollyReqs,
                    state=ctr.State.READY,
                    attrs=dict(target_id=env_id)
                ).order(-sdk2.Resource.id).first()

                if not binary_resource:
                    continue

                logging.info("ReqMiner resulting resource found: {}".format(binary_resource.id))

                subtask = MollyRun(
                    self,
                    description="Running Molly Scan for {} (subtask of {})".format(env_id, self.id),
                    owner=self.owner
                )
                subtask.max_restarts = 2
                subtask.Parameters.target_uri = item.get('url')
                subtask.Parameters.profile = "Yandex"

                if item.get('users'):
                    subtask.Parameters.users = ','.join(item.get('users'))

                if item.get('target'):
                    subtask.Parameters.target = item.get('target')

                if item.get('st_queue'):
                    subtask.Parameters.st_queue = item.get('st_queue')

                if item.get('max_rps'):
                    subtask.Parameters.rps = item.get('max_rps')

                if item.get('ignore_time_limit'):
                    subtask.Parameters.ignore_time_limit = True

                subtask.Parameters.request_samples_resource = 'sandbox-resource:{}'.format(binary_resource.id)
                subtask.Parameters.request_samples_resource_type = self.FMT_JSON2

                subtask.Parameters.ignore_time_limit = False
                subtask.Parameters.auto_tickets = True

                logging.info("Saving subtask")
                subtask.save()
                children.append(subtask)

            logging.info("Waiting for MOLLY_RUN subtasks to finish")
            raise sdk2.WaitTask([c.enqueue() for c in children], list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)
