#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import ipaddress
import requests
import os
import sys

try:
    from aggregator.yt_log import BalancerAggregator
    from generator.molly import ReqGenerator
    from utils.fs_utils import create_json_file

except ImportError:

    from sandbox.projects.Molly.MollyReqMiner.aggregator.yt_log import BalancerAggregator
    from sandbox.projects.Molly.MollyReqMiner.generator.molly import ReqGenerator
    from sandbox.projects.Molly.MollyReqMiner.utils.fs_utils import create_json_file


def resolve_filetered_nets(net_type):
    pref_net = {}
    types = {'v4', 'v6'}
    prefs = set()
    if net_type not in types:
        return None, prefs, pref_net

    if net_type == 'v4':
        # INET_ADDRSTRLEN
        short = 16
        res_macr_url = 'https://racktables.yandex.net/export/expand-fw-macro.php?macro=_MOLLY_AGENTS_TUN64_NETS_'

    else:
        # INET6_ADDRSTRLEN
        short = 46
        res_macr_url = 'https://racktables.yandex.net/export/expand-fw-macro.php?macro=_SECURITY_MTN_NETS_'

    try:
        nets = requests.get(res_macr_url).text.split('\n')

    except requests.exceptions.ConnectionError:
        print('Could not connect to the racktables')
        return

    for sub in nets:
        try:
            if net_type == 'v4':
                ipnet = ipaddress.IPv4Network(sub)

            else:
                ipnet = ipaddress.IPv6Network(sub)

        except ipaddress.AddressValueError:
            print('Could not construct ipv6 subnet info for : {} '.format(sub))
            continue

        pref = sub.split('/')[0]
        if net_type == 'v4':
            pref = '.'.join(pref.split('.')[:-2])

        else:
            pref = ':'.join(pref.split(':')[:-2])

        plen = len(pref)
        if plen < short:
            short = plen

        pref_net[pref] = ipnet

    for pref, ipnet in pref_net.items():
        new_pref = pref[:short]
        prefs.add(new_pref)
        pref_net[new_pref] = ipnet

    return short, prefs, pref_net


def ip_filter():
    sh_v4, prs_v4, nets4 = resolve_filetered_nets('v4')
    _, prs_v6, nets6 = resolve_filetered_nets('v6')
    prs_v4 |= prs_v6
    nets4.update(nets6)

    return sh_v4, prs_v4, nets4


if __name__ == '__main__':
    service_id = sys.argv[1]
    target_hostname = None
    try:
        target_hostname = sys.argv[2]
    except Exception as e:
        pass
    start_datetime = datetime.datetime.now() - datetime.timedelta(days=30)
    end_datetime = datetime.datetime.now()
    token = os.environ['YT_TOKEN']
    log_parser = BalancerAggregator(yt_server='hahn.yt.yandex.net', ip_filter_data=ip_filter(),
                                    filter_param='everybodybecoolthisis')
    log_parser.aggregate_path_params(service_id, start_datetime, end_datetime)
    req_gen = ReqGenerator(yt_server='hahn.yt.yandex.net', yt_token=token, target_hostname=target_hostname)
    req_gen.generate_molly_reqs(service_id)
    items = req_gen.get_molly_reqs()
    create_json_file('requests.json', items)
