#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import json
import yt.wrapper as yt

try:
    from utils import yt_utils as utils

except ImportError:
    from sandbox.projects.Molly.MollyReqMiner.utils import yt_utils as utils


class ReqGenerator(utils.YtLogParser):
    """Parse aggregator results and generate molly request samples"""

    def __init__(self, yt_server='hahn.yt.yandex.net', yt_token=None, aggr_table=None, molly_table=None,
                 target_hostname=None):
        """Sets standart parameters for yt providers
        :param yt_server:
        :param aggr_table:
        :param molly_table:
        """
        super(ReqGenerator, self).__init__(yt_server, yt_token)
        self.utils = utils
        self.aggr_table = aggr_table
        self.molly_req_table = molly_table
        self.service = None
        self.target_hostname = target_hostname

    @staticmethod
    def _make_sep_params(params):
        query = ''
        first = True
        for name, value in params.items():
            if first:
                query_part = name + '=' + value
                first = False

            else:
                query_part = '&' + name + '=' + value

            query += query_part

        return query

    @staticmethod
    def _encode_multipart(ctype, params):
        boundary = ctype.split(';')[1][1:].split('=')[1]
        body = '--{}\r\n'.format(boundary)
        for name, param_data in params.items():
            filename = param_data[0]
            if isinstance(filename, unicode):
                filename = filename.encode('utf-8')
            value = param_data[1]
            if isinstance(value, unicode):
                value = value.encode('utf-8')

            if not filename:
                body += 'Content-Disposition: form-data; name="{}"\r\n'.format(name)
            else:
                body += 'Content-Disposition: form-data; name="{}"; filename="{}"\r\n'.format(name, filename)

            body += '\r\n'
            body += value
            body += '\r\n'
            body += '--{}\r\n'.format(boundary)

        return body

    @staticmethod
    def _construct_cookie_string(cookies):
        string = ''
        last_num = len(cookies) - 1
        num = 0
        for key, value in cookies.items():
            string += key + '=' + value
            if num != last_num:
                string += '; '
                num += 1

        return string

    def _make_headers(self, hosts, cookies, ctype):
        headers = [{'Name': 'Host', 'Value': hosts[0]}]
        if ctype:
            headers.append({'Name': 'Content-Type', 'Value': ctype})

        if cookies:
            headers.append({'Name': 'Cookie', 'Value': self._construct_cookie_string(cookies)})

        return headers

    def _make_typed_body(self, ctype, params):
        body = ''
        if ctype.startswith('application/x-www-form-urlencoded'):
            body = self._make_sep_params(params)

        elif ctype.startswith('multipart/form-data'):
            body = self._encode_multipart(ctype, params)

        elif ctype == 'application/json':
            try:
                body = json.dumps(params)

            except UnicodeDecodeError:
                return body

        return body

    @yt.aggregator
    def _yt_generate_requests(self, recs):
        for rec in recs:
            ctype = None
            body = ''
            raw_query = ''
            method = rec.get('method')
            path = rec.get('path')
            prot = rec.get('protocol')
            hosts = rec.get('hosts')
            params = json.loads(rec.get('all_params'))
            cookies = json.loads(rec.get('all_cookies'))

            if method == 'GET':
                raw_query = self._make_sep_params(params)

            if method in {'PATCH', 'POST', 'PUT'}:
                ctype = rec.get('ctype')
                body = self._make_typed_body(ctype, params)

            if self.target_hostname:
                hosts = [self.target_hostname]

            headers = self._make_headers(hosts, cookies, ctype)
            yield {'method': method, 'path': path, 'rawquery': raw_query, 'proto': prot, 'body': body,
                   'headers': headers}

    def generate_molly_reqs(self, service_id):
        if not self.aggr_table:
            self.aggr_table = '//tmp/{0}_molly_req_aggr.{1}'.format(service_id,
                                                                    datetime.datetime.now().strftime('%Y-%m-%d'))

        if not self.molly_req_table:
            self.molly_req_table = '//tmp/{0}_molly_request.{1}'.format(service_id,
                                                                        datetime.datetime.now().strftime('%Y-%m-%d'))

        self.yt.run_map(self._yt_generate_requests, self.aggr_table, self.molly_req_table, format=self.yt.YsonFormat())

    def get_molly_reqs(self):
        items = []
        for rec in self.yt.read_table(self.molly_req_table, format=self.yt.JsonFormat()):
            items.append(rec)

        return items
