# coding: utf-8

import logging
import json
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from molly import Molly


log = logging.getLogger(__name__)


class SeverityLevel(sdk2.parameters.String):
    name = "severity_level"
    description = "Minimum severity level to mark target as vulnerable"
    required = True

    SEVERITY_IGNORE = -1
    SEVERITY_UNKNOWN = 0
    SEVERITY_INFORMATION = 10
    SEVERITY_LOW = 20
    SEVERITY_MEDIUM = 30
    SEVERITY_HIGH = 40
    SEVERITY_CRITICAL = 50
    SEVERITY_BLOCKER = 60
    SEVERITIES = (
        ("Not informative", SEVERITY_IGNORE),
        ("Unknown", SEVERITY_UNKNOWN),
        ("Information", SEVERITY_INFORMATION),
        ("Low", SEVERITY_LOW),
        ("Medium", SEVERITY_MEDIUM),
        ("High", SEVERITY_HIGH),
        ("Critical", SEVERITY_CRITICAL),
        ("Blocker", SEVERITY_BLOCKER),
    )
    choices = SEVERITIES
    default_value = SEVERITY_MEDIUM


class RequestSamplesResourceType(sdk2.parameters.String):
    name = "samples_resource_format"
    description = "Sandbox resource HTTP request samples format"
    required = False

    FMT_JSON = 0
    FMT_BURP = 1
    FMT_SERP = 2
    FMT_SERP2 = 3
    FMT_JSON2 = 4
    SAMPLE_FORMATS = (
        ("Molly-compatible format", FMT_JSON),
        ("Burp XML", FMT_BURP),
        ("SERP Sandbox", FMT_SERP),
        ("SERP Sandbox (short)", FMT_SERP2),
        ("Molly-preconverted JSON format", FMT_JSON2),
    )
    choices = SAMPLE_FORMATS
    default_value = FMT_SERP2


class MollyScanReport(sdk2.Resource):
    any_arch = True


class MollyRun(sdk2.Task):
    MOLLY_BASE = "https://molly.yandex-team.ru/api"

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        target_uri = sdk2.parameters.String("Target URI", required=True)
        profile = sdk2.parameters.String("Scan profile", default="Yandex", required=True)

        user_agent = sdk2.parameters.String("User-Agent", required=False)
        auth_profile = sdk2.parameters.String("Authentication profile UID", required=False)
        qs_params = sdk2.parameters.String("QueryString params", required=False)
        request_samples = sdk2.parameters.String("Request samples UID", required=False)
        request_samples_resource = sdk2.parameters.String("Sandbox resource id containing HTTP request samples", required=False)
        request_samples_resource_type = RequestSamplesResourceType()

        abc_id = sdk2.parameters.String("ABC id", required=False)
        users = sdk2.parameters.String("Notify users", required=False)
        target = sdk2.parameters.String("Aggregation Target", required=False)
        st_queue = sdk2.parameters.String("Tracker Queue", required=False)
        rps = sdk2.parameters.String("RPS limit", required=False)

        ignore_time_limit = sdk2.parameters.Bool("Ignore scan time limit", default=False, required=False)
        ticket_id = sdk2.parameters.String("Report to ticket", required=False)

        auto_tickets = sdk2.parameters.Bool("Create tickets for vulnerabilities with specified severity level", default=True, required=False)

        severity = SeverityLevel()
        aggregate_uid = sdk2.parameters.String("Use specified samples aggregate id", required=False)

        with sdk2.parameters.Output:
            status = sdk2.parameters.Bool(
                "Did scan found any new vulnerabilities with target severity level",
                default=False, required=True
            )
            report = sdk2.parameters.String(
                "Link to resulting report", default="", required=False
            )

    def on_execute(self):
        vault = sdk2.Vault.data(self.owner, "MOLLY_SANDBOX_TOKEN")
        ctx = Molly(self.MOLLY_BASE, vault)

        log.info("Starting task")
        scan_params = {"profile": self.Parameters.profile, 'severity': self.Parameters.severity}
        if self.Parameters.user_agent:
            scan_params["user_agent"] = self.Parameters.user_agent
        if self.Parameters.auth_profile:
            scan_params["auth_profile"] = self.Parameters.auth_profile
        if self.Parameters.qs_params:
            scan_params["qs_params"] = self.Parameters.qs_params
        if self.Parameters.request_samples:
            scan_params["request_samples"] = self.Parameters.request_samples
        if self.Parameters.abc_id:
            scan_params["abc_id"] = self.Parameters.abc_id
        if self.Parameters.users:
            scan_params["users"] = self.Parameters.users
        if self.Parameters.target:
            scan_params["target"] = self.Parameters.target
        if self.Parameters.st_queue:
            scan_params["st_queue"] = self.Parameters.st_queue
        if self.Parameters.rps:
            scan_params["rps"] = self.Parameters.rps
        if self.Parameters.ticket_id:
            scan_params["st_ticket"] = self.Parameters.ticket_id
        if self.Parameters.ignore_time_limit:
            scan_params["ignore_time_limit"] = self.Parameters.ignore_time_limit
        if self.Parameters.request_samples_resource:
            scan_params["request_samples_resource"] = self.Parameters.request_samples_resource
        if self.Parameters.request_samples_resource_type:
            scan_params["request_samples_resource_type"] = self.Parameters.request_samples_resource_type
        if not self.Parameters.auto_tickets:
            scan_params["no_auto_tickets"] = True
        if self.Parameters.aggregate_uid:
            scan_params["request_samples_aggregate_uid"] = self.Parameters.aggregate_uid

        if not ctx.init_scan(self.Parameters.target_uri, **scan_params):
            raise TaskFailure("Molly scan init failed")

        self.Parameters.report = "https://molly.yandex-team.ru/report/{}".format(ctx.scan_id)
        target_severity_level = int(self.Parameters.severity)

        result = ctx.fetch_scan_progress()
        log.info(json.dumps(result))

        while result.get("status", "error") == "in_progress":
            time.sleep(30)
            result = ctx.fetch_scan_progress()

        log.debug(json.dumps(result))

        if result.get("status", "error") == "error":
            raise TaskFailure('Molly scan failed! Scan id: {}'.format(ctx.scan_id))

        resource = sdk2.ResourceData(MollyScanReport(self, "Report JSON", "report.json"))
        resource.path.write_bytes(json.dumps(result))
        resource.ready()

        is_vulnerable = False
        for vuln in result.get("vulnerabilities", []):
            if vuln.get("is_triaged", False) or vuln.get("is_false_positive", False):
                continue
            if int(vuln.get("severity_level", -1)) >= target_severity_level:
                is_vulnerable = True
                break
        self.Parameters.status = is_vulnerable
