# -*- coding: utf-8 -*-

import logging
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters

import requests
import json
import time


class Task(sdk_task.SandboxTask):
    """ Task which only gets data from rtmr and plot somewhere """

    type = "MULTIMEDIA_RTMR_MONITOR"

    class DBName(sdk_parameters.SandboxStringParameter):
        name = "database_name"
        description = "Name of grafana database"
        default_value = "dev-ttl-30d"

    class SeriesName(sdk_parameters.SandboxStringParameter):
        name = "series_name"
        description = "Name of grafana series"
        default_value = "multimedia_rtmr_monitor_test"

    input_parameters = [DBName, SeriesName]

    table_name = "multimedia_monitor"
    api_url = "http://rtmr.search.yandex.net:8080/api/v1/tables.plain"
    keys_per_request = 20

    grafana_dbname = None
    grafana_series = None
    grafana_url = "http://influxdb00.haze.yandex.net:8086/write?db={db}"
    grafana_series_tmpl_old = "{series},tld={tld},service={service},platform={platform},intent={intent}"
    grafana_series_tmpl = "{series}_{service},tld={tld},platform={platform},intent={intent}"

    services = ["video", "images", "web"]
    regions = ["ru", "kz", "ua", "tr", "by"]
    platforms = ["desktop", "pad", "touch", "mobile"]
    intents = ["total", "quick", "fresh", "ultra", "cm2", "is_vcomm", "porno", "vserial"]
    keys = ["click", "request", "click/to_hosting", "click/vc"
            "duration", "heartbeat", "playing_duration", "related_request",
            "duration/vc", "heartbeat/vc", "playing_duration/vc",
            "click/greenurl", "click", "click/greenurl_comm", "click/thumb/normal",
            "click/commercial",
            "request", "results/related_query",
            "wizard_show/images", "wizard_show/video",
            "click/wiz/serp", "click/wiz/vc"]

    aliases = {
        "web": {
            "wizard_show_images": [("images", "wizard_shows")],
            "wizard_show_video": [("video", "wizard_shows")],
            "request": [("images", "web_requests"), ("video", "web_requests")]
        }
    }

    def add_suffix(self, prefixes, suffixes):
        res = []

        for v in prefixes:
            map(lambda x: res.append(v + "/" + x), suffixes)

        return res

    def generate_requests(self):
        keys = [""]
        keys = self.add_suffix(keys, self.services)
        keys = self.add_suffix(keys, self.regions)
        keys = self.add_suffix(keys, self.platforms)
        keys = self.add_suffix(keys, self.intents)
        keys = self.add_suffix(keys, self.keys)

        requests = map(lambda x: json.dumps({"Table": self.table_name, "Key": x}), keys)
        return requests

    def parse_results(self, text):
        res = []
        for l in (l.split('\t') for l in text.split('\n')):
            if len(l) < 2:
                continue
            res.append(l[0:2])
            logging.debug('key={} value={}'.format(l[0], l[1]))

        return res

    def request(self, requester, url, retries=3):
        r = None
        for i in range(retries):
            try:
                r = requester(url)
                break
            except requests.exceptions.ConnectionError, e:
                logging.error("Exception. Try number{}".format(i))
                if i == retries - 1:
                    raise e
                time.sleep(60)

        if r.status_code != 200 and r.status_code != 204:
            raise Exception("Unknown error: code={} url={}".format(r.status_code, r.url))

        return r

    def process_requests(self, urls):
        results = {}
        for n in range(len(urls)/self.keys_per_request + 1):
            b = n * self.keys_per_request
            e = b + self.keys_per_request
            params = {"request": urls[b:e]}

            l = lambda x: requests.get(x, params=params)
            r = self.request(l, self.api_url)

            for k, v in self.parse_results(r.text):
                results[k] = results.get(k, 0.) + float(v)

        return results

    def fill_data(self, tmpl, indata):
        data = u''
        for k, v in indata.iteritems():
            k = json.loads(k)
            for x in v:
                data += self.grafana_series_tmpl.format(series=self.grafana_series, **k)
                data += " {}={}\n".format(x[0], x[1])

        return data

    def push_results(self, results):
        tmp_data = {}
        for k, v in results.iteritems():
            kk = k.split("/")[1:]
            if len(kk) < 5:
                continue

            key = {"service": kk[0], "tld": kk[1], "platform": kk[2], "intent": kk[3]}
            value = ('_'.join(kk[4:]), v)
            self.push_data(tmp_data, key, value)
            self.apply_aliases(tmp_data, key, value)

        url = self.grafana_url.format(db=self.grafana_dbname)

        data = self.fill_data(self.grafana_series_tmpl, tmp_data)
        l = lambda x: requests.post(x, data=data[:-1])
        self.request(l, url)

        data = self.fill_data(self.grafana_series_tmpl_old, tmp_data)
        l = lambda x: requests.post(x, data=data[:-1])
        self.request(l, url)

    def apply_aliases(self, data, key, value):
        service = key["service"]
        if service not in self.aliases:
            return
        aliases = self.aliases[service]
        metric = value[0]
        if metric not in aliases:
            return
        for a in aliases[metric]:
            new_key = dict(key)
            new_key["service"] = a[0]
            new_value = (a[1], value[1])
            self.push_data(data, new_key, new_value)

    def push_data(self, data, key, value):
        data_key = json.dumps(key)

        if data_key not in data:
            data[data_key] = []
        data[data_key].append(value)

    def on_execute(self):
        self.grafana_dbname = self.ctx.get(self.DBName.name)
        self.grafana_series = self.ctx.get(self.SeriesName.name)
        logging.info("Grafana db={} series={}".format(self.grafana_dbname, self.grafana_series))

        urls = self.generate_requests()
        results = self.process_requests(urls)
        self.push_results(results)


__Task__ = Task
