# -*- coding: utf-8 -*-

from sandbox.projects.common.NannyTask import NannyTask
from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter


class ServiceName(SandboxStringParameter):
    name = 'nanny_service_name'
    description = 'Nanny search base download service name'
    required = True


class ShardState(SandboxStringParameter):
    name = 'shard_state'
    description = 'Shard state'
    required = True


class RequiredShardState(SandboxStringParameter):
    name = 'required_shard_state'
    description = 'Required shard state'
    required = False


class ShardName(SandboxStringParameter):
    name = 'shard_name'
    description = 'Shard name (i.e. imgsth, imgidx)'
    required = False


class ShardGeneration(SandboxStringParameter):
    name = 'shard_generation'
    description = 'Shard generation number'
    required = False


class MakeBsconfigShard(SandboxBoolParameter):
    name = 'make_bsconfig_shard'
    description = 'Make and register bsconfig shard'
    required = False
    default_value = True


class VaultKey(SandboxStringParameter):
    name = 'vault_key'
    description = 'Sandbox vault name'
    required = True


class WaitActiveStateBeforeActivate(SandboxBoolParameter):
    name = 'wait_active_state_before_activate'
    description = 'Wait active state before activate'
    required = False
    default_value = True


class WaitActivateCompetion(SandboxBoolParameter):
    name = 'wait_activate_completion'
    description = 'Wait activation completion'
    required = False
    default_value = True


class MapreduceServer(SandboxStringParameter):
    name = 'server'
    description = 'Mapreduce server to download base'
    required = False


class MapreduceBasePrefix(SandboxStringParameter):
    name = 'prefix'
    description = 'Mapreduce path prefix of base'
    required = False
    default_value = 'images'


class NannyBuildBase(NannyTask):
    """
        Запуск процесса подготовки поисковой базы в nanny
    """

    type = 'NANNY_BUILD_BASE'

    cores = 1

    input_parameters = [ServiceName, MapreduceServer, MapreduceBasePrefix, ShardState,
                        RequiredShardState, ShardName, ShardGeneration, MakeBsconfigShard,
                        VaultKey, WaitActiveStateBeforeActivate, WaitActivateCompetion]

    def _deploy(self, client, service_name):
        runtime_attrs = client.get_service_runtime_attrs(service_name)
        instances = runtime_attrs['content']['instances']

        keys = [
            ShardState.name,
            RequiredShardState.name,
            ShardName.name,
            ShardGeneration.name,
            MakeBsconfigShard.name,
            MapreduceServer.name,
            MapreduceBasePrefix.name,
        ]

        for group in instances['extended_gencfg_groups']['groups']:
            new_tags = []
            for key in keys:
                value = self.ctx.get(key)
                if value:
                    # Make 1 or 0 for boolean values
                    value = '1' if value is True else '0' if value is False else value

                    new_tags.append('%s=%s' % (key, value))

            group['tags'] = new_tags

        result = client.update_service_instances(service_name, {
            'content': instances,
            'comment': 'Start MR base downloading from sandbox',
            'snapshot_id': runtime_attrs['_id']
        })

        return client._deploy(service_name, result['runtime_attrs']['_id'])

    def on_execute(self):
        vault_key = self.ctx.get(VaultKey.name)
        client = NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data(vault_key, 'nanny-oauth-token')
        )

        service_name = self.ctx.get(ServiceName.name)
        wait_before_activate = bool(self.ctx.get(WaitActiveStateBeforeActivate.name))
        wait_completion = bool(self.ctx.get(WaitActivateCompetion.name))

        if wait_before_activate and not self._all_tasks_completed():
            self._wait_all_tasks(client, service_name)

        if 'new_nanny_snapshot' not in self.ctx:
            task = self._deploy(client, service_name)
            self.ctx['new_nanny_snapshot'] = task[u'set_target_state'][u'snapshot_id']

        if 'new_nanny_snapshot' in self.ctx and wait_completion:
            if not self._wait_current_task(
                    client, service_name, self.ctx['new_nanny_snapshot']):
                self.wait_time(60)

    @property
    def footer(self):
        return '<h4>Download service: <a href="https://nanny.yandex-team.ru/'\
               'ui/#/services/catalog/{service_name}/">{service_name}</a></h4>'.\
               format(service_name=self.ctx.get(ServiceName.name))


__Task__ = NannyBuildBase
