# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, SandboxIntegerParameter

from sandbox.projects.MediaLib import monitor, monitoring_input_parameters
from sandbox.projects.common.nanny import nanny
# from projects.common.decorators import retries
from sandbox.projects.common import utils

logger = logging.getLogger()

RETRIES = 10
DELAY = 30

# total service statuses
SERVICE_SUMMARY_ACTIVE_STATE = (u'ONLINE', )

# states for snapshots
ACTIVE_STATES = (u'ACTIVE', u'ACTIVATING')
NON_ACTIVE_STATES = (u'PREPARED', u'DEACTIVATE_PENDING', u'GENERATING', u'CREATED', u'DEACTIVATING', u'PREPARING')


class DashboardName(SandboxStringParameter):
    name = 'dashboard_name'
    description = 'Dashboard name for find snapshots for deleting'
    required = True


class DeleteResourceTaskId(SandboxStringParameter):
    name = 'delete_resource_task_id'
    description = 'Resource task id for deleting snapshots'
    required = True


class RollbackActivateRecipe(SandboxStringParameter):
    name = 'rollback_activate_recipe'
    description = 'Recipe for activating rollbacking services in dashboard. If not set will not remove active'
    default_value = ''


class IsShardmap(SandboxBoolParameter):
    name = 'is_shardmap'
    description = 'Is Resource shardmap?'
    default_value = True


class WaitCheck(SandboxIntegerParameter):
    name = 'wait_check'
    description = 'Период опроса состояния (секунды)'
    default_value = 120


class NannyDeleteSnapshotResource(SandboxTask):
    """Delete services snapshots by resource task id"""

    type = 'NANNY_DELETE_SNAPSHOT_RESOURCE'

    execution_space = 1024
    # se_tags = {'limit1': 1}

    input_parameters = (DashboardName, DeleteResourceTaskId, RollbackActivateRecipe, IsShardmap,
                        WaitCheck) + monitoring_input_parameters

    def initCtx(self):
        self.ctx['kill_timeout'] = 27 * 60 * 60

    def generate_snapshot_list(self, nanny_client, services, delete_resource_task_id, is_shardmap):
        # dict of snapshots for delete
        # remove_snapshots[remove_snapshot] = service
        remove_snapshots = {}

        # dict of rollback shapshots for services
        # rollback_snapshots[service] = rollback_snapshot
        rollback_snapshots = {}

        # Find snapshots to delete and rollback snapshots
        for service in services:
            logger.debug("service: {}".format(service))

            found_remove_snapshot = False

            service_state = nanny_client.get_service_current_state(service)
            logger.debug("service_state: {}".format(service_state))

            for snapshot in service_state[u'content'][u'active_snapshots']:
                logger.debug("service: {} snapshot: {}".format(service, snapshot))

                # get runtime attrs
                snapshot_id = snapshot[u'snapshot_id']
                runtime_attrs = nanny_client.get_history_runtime_attrs(snapshot_id)
                logger.debug("snapshot: {} runtime_attrs: {}".format(snapshot, runtime_attrs))

                if is_shardmap:
                    service_shardmap_dump = runtime_attrs[u'content'][u'resources'].get(u'sandbox_bsc_shard')
                    if service_shardmap_dump is None:
                        raise Exception(
                            "get_shardmap_name sandbox_bsc_shard is absent but you set shardmap flag."
                        )

                    if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
                        shardmap_task_id = service_shardmap_dump[u'sandbox_shardmap'][u'task_id']
                        if int(shardmap_task_id) == int(delete_resource_task_id):
                            # delete this snapshot
                            found_remove_snapshot = True
                            remove_snapshots[snapshot_id] = service
                            logger.debug('add remove_snapshots[{}] = {}'.format(snapshot_id, service))
                        elif not rollback_snapshots.get(service):
                            # seve rollback snapshot
                            rollback_snapshots[service] = snapshot_id
                            logger.debug('add rollback_snapshots[{}] = {}'.format(service, snapshot_id))
                    else:
                        raise Exception(
                            "get_shardmap_name has got wrong chosen_type (differ from SANDBOX_SHARDMAP): {}".format(runtime_attrs)
                        )
                else:
                    # other resource
                    logger.debug("Trying to delete resource that isn't a shardmap")
                    raise NotImplementedError('Implement this feature')

            # alert for not founding snapshot for deleting in service
            if not found_remove_snapshot:
                msg = 'Has not been found snapshot for deleting in the service: {}'.format(service)
                logger.debug(msg)
                self.set_info(msg)

        return remove_snapshots, rollback_snapshots

    @monitor
    def on_execute(self):
        dashboard_name = self.ctx.get(DashboardName.name, None)
        delete_resource_task_id = self.ctx.get(DeleteResourceTaskId.name)
        wait_check = utils.get_or_default(self.ctx, WaitCheck)
        is_shardmap = utils.get_or_default(self.ctx, IsShardmap)
        rollback_actiavte_recipe = utils.get_or_default(self.ctx, RollbackActivateRecipe)

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        services = nanny_client.get_dashboard_services(dashboard_name)
        logger.debug("Services: {}".format(services))

        # 1. Rollbacking active and delete other snapshots
        if not self.ctx.get('finish_found_snapshots'):

            # 1.1 Generate list of snapshots to delete
            self.set_info("Genetate snapshots for deleting.")
            remove_snapshots, rollback_snapshots = self.generate_snapshot_list(nanny_client, services, delete_resource_task_id, is_shardmap)

            if not remove_snapshots:
                logger.debug("Nothing to delete: remove_snapshots is empty")
                raise Exception("Nothing to delete: remove_snapshots is empty")

            # 1.2 Start deleting and rollbacking
            logger.debug("Start removing snapshots: {}".format(remove_snapshots))
            self.set_info("Start deleting snapshots.")

            for remove_snapshot in remove_snapshots.keys():
                service = remove_snapshots[remove_snapshot]
                snapshot_state = nanny_client.get_history_runtime_attrs_states(service, remove_snapshot)
                snapshot_current_state = snapshot_state['current_state']['state']
                logger.debug(
                    "Get remove snapshot: {} service: {} snapshot_state: {}".format(remove_snapshot, service, snapshot_state)
                )

                # flag
                wait_rollbacked = False

                # Rollback ACTIVE
                if snapshot_current_state in ACTIVE_STATES:
                    # only if we have got rollback_actiavte_recipe
                    if rollback_actiavte_recipe:
                        msg = 'Send rollback comand for service: {} snapshot: {}'.format(service, remove_snapshot)
                        logger.debug(msg)
                        self.set_info(msg)
                        # rollback
                        rollback_snapshot = rollback_snapshots[service]
                        resp = nanny_client.set_snapshot_state(
                            service,
                            rollback_snapshot,
                            'ACTIVE',
                            'rollback from {} to {} by DeleteMediaResource'.format(remove_snapshot, rollback_snapshot),
                            recipe=rollback_actiavte_recipe,
                            set_as_current=True,
                        )
                        logger.debug('resp: %s', resp)
                        wait_rollbacked = True
                    else:
                        msg = 'Skip snapshot from deleting: {} service: {}'.format(remove_snapshot, service)
                        logger.debug(msg)
                        self.set_info(msg)

                # Delete NON_ACTIVE_STATES snapshots
                elif snapshot_current_state in NON_ACTIVE_STATES:
                    # delete immidietly
                    msg = 'Send deleting comand for service: {} snapshot: {}'.format(service, remove_snapshot)
                    logger.debug(msg)
                    self.set_info(msg)
                    resp = nanny_client.set_snapshot_state(
                        service,
                        remove_snapshot,
                        'DESTROYED',
                        'deleted snapshot: {} by DeleteMediaResource'.format(remove_snapshot)
                    )
                    logger.debug('resp: %s', resp)

                # Other states: e.g. DESTROYING
                else:
                    logger.debug(
                        "Snapshot {} is in REMOVING/STRANGE state: for service: {} сurrent_state: {}".format(
                            remove_snapshot, service, snapshot_current_state
                        )
                    )

                # set ctx
                self.ctx['wait_rollbacked'] = wait_rollbacked

            # 1.3 Restart task while rollbacked
            if self.ctx.get('wait_rollbacked'):
                self.wait_time(wait_check)

            # 1.4 Stop finding snapshots for deleting
            self.ctx['finish_found_snapshots'] = True
            self.set_info("Snapshots have been deleted.")
            self.set_info("Waiting for services get ACTIVE state.")

        # 2. Checks status of all services
        # ./nanny/src/nanny/model/docs/states.py
        all_active = True
        for service in services:
            logger.debug("ckeck status service: {}".format(service))

            service_state = nanny_client.get_service_current_state(service)
            state = service_state[u'content'][u'summary'][u'value']
            logger.debug("ckeck status service: {} state: {}".format(service, state))

            if state not in SERVICE_SUMMARY_ACTIVE_STATE:
                logger.debug("ckeck status service not in SERVICE_SUMMARY_ACTIVE_STATE: {} state: {}".format(service, state))
                all_active = False
                break

        if not all_active:
            self.wait_time(wait_check)

        self.set_info("All services are in ACTIVE state")


__Task__ = NannyDeleteSnapshotResource
