# - * - encoding: UTF-8 - * -

from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import binary_task


class NewsExpChecker(binary_task.LastBinaryTaskRelease, sdk2.Task):
    CHECK_TIME_SEC = 120
    CHECK_PERIOD_SEC = 10
    TOP_SIZE = 5

    def load_json(self, url):
        import requests
        request = requests.get(url)

        import json
        return json.loads(request.text)

    def parse_main_data(self, url, return_queue):
        data = self.load_json(url)
        timestamp = int(data['Topnews']['iter_timestamp'])
        top_ids = [None] * self.TOP_SIZE
        for top in data['Topnews']['tabs']:
            if top['idx'] == 0:
                for top_element in top['news']:
                    if top_element['i'] <= self.TOP_SIZE:
                        top_ids[top_element['i'] - 1] = top_element['persistent_id']
                break
        return_queue.put(['main', timestamp, top_ids])

    def parse_news_data(self, url, return_queue):
        data = self.load_json(url)
        timestamp = None
        top_ids = []
        for element in data['NEWS_RUBRIC']['NEWS_SLAVE_NEWSD']['response']:
            if element.get('handler') == 'state_time' and \
            element.get('mode') == 'fresh' and \
            element.get('type') == 'news_newsd_response':
                timestamp = element['data']
            if 'data' in element and isinstance(element['data'], dict) and 'tops' in element['data']:
                for top in element['data']['tops']:
                    if top['rubric']['id'] == 0:
                        for top_element in top['stories'][:self.TOP_SIZE]:
                            top_ids.append(top_element['annot']['persistent_id'])
                        break
        return_queue.put(['news', timestamp, top_ids])

    def get_persistent_id_from_story_url(self, url):
        from urlparse import urlparse, parse_qs
        return parse_qs(urlparse(url).query)['persistent_id'][0]

    def parse_tops_data(self, url, domain, return_queue):
        data = self.load_json(url)
        timestamp = data['iter_time']
        top_ids = []
        for rubric in data['data'][domain]['ru']['rubrics']:
            if rubric['alias'] == 'index':
                for top_element in rubric['stories'][:self.TOP_SIZE]:
                    top_ids.append(self.get_persistent_id_from_story_url(top_element['url']))
                break
        return_queue.put(['tops', timestamp, top_ids])

    def print_ids_difference(self, first_ids, second_ids, first_url, second_url):
        import logging
        logging.info("\tDifference in: ")
        logging.info(first_url)
        logging.info(second_url)
        for i in range(len(first_ids)):
            if first_ids[i] != second_ids[i]:
                logging.info('\t' + str(i + 1) + '. ' + str(first_ids[i]) + ' / ' + str(second_ids[i]))

    def print_difference(self, main, news, tops, tops_available, main_url, news_url, tops_url):
        no_common_timestamp = True
        for timestamp, ids in main.items():
            if timestamp in news and (not tops_available or timestamp in tops):
                no_common_timestamp = False
            if timestamp in news and news.get(timestamp) != ids:
                self.print_ids_difference(ids, news.get(timestamp), main_url, news_url)
            elif tops_available and timestamp in tops and tops.get(timestamp) != ids:
                self.print_ids_difference(ids, tops.get(timestamp), main_url, tops_url)

        import logging
        if no_common_timestamp:
            logging.info('\tNo common timestamp')
        logging.info('\n')

    def check(self, main, news, tops, tops_available):
        import logging
        logging.info("all_main " + str(main))
        logging.info("all_news " + str(news))
        logging.info("all_tops " + str(tops))

        for timestamp, ids in main.items():
            if news.get(timestamp) == ids and (not tops_available or tops.get(timestamp) == ids):
                return True
        return False

    def fill_tops_data(self, return_queue, main, news, tops):
        while not return_queue.empty():
            name, timestamp, ids = return_queue.get()
            if name == 'main':
                main[timestamp] = ids
            elif name == 'news':
                news[timestamp] = ids
            else:
                tops[timestamp] = ids

    def check_domain(self, domain):
        import logging
        logging.info("Start checking domain: " + domain)
        test_id = self.Parameters.test_id

        no_rotation_test_id = 'test-id=457867'
        main_url = 'https://yandex.' + domain + '/?' + no_rotation_test_id + '_' + test_id + '&cleanvars=topnews&client=news-exp-checker'
        news_url = 'https://yandex.' + domain + '/news/?' + no_rotation_test_id + '_' + test_id + '&json_dump_responses=NEWS_SLAVE_NEWSD&client=news-exp-checker'
        tops_available = self.Parameters.rank_extra != ''
        tops_url = None

        if tops_available:
            rank_extra = self.Parameters.rank_extra
            disable_rotation_flags = 'flags=yxnews_rotate_related_stories_russian_top=0;\
                                            yxnews_rotate_related_stories_regions_tops=0;\
                                            yxnews_rotate_related_stories_index_tops=0;\
                                            yxnews_disable_rotation=1'
            tops_url = 'https://news.yandex.' + domain + '/api/v2/tops_export_all?' + disable_rotation_flags + '&circuit=' + rank_extra

        main = {}
        news = {}
        tops = {}
        processes = []

        from multiprocessing import Process, Queue
        return_queue = Queue()

        import time
        start_time = time.time()
        while time.time() - start_time < self.CHECK_TIME_SEC:
            main_process = Process(target=self.parse_main_data, args=(main_url, return_queue))
            main_process.start()
            processes.append(main_process)
            news_process = Process(target=self.parse_news_data, args=(news_url, return_queue))
            news_process.start()
            processes.append(news_process)
            if tops_available:
                tops_process = Process(target=self.parse_tops_data, args=(tops_url, domain, return_queue))
                tops_process.start()
                processes.append(tops_process)
            self.fill_tops_data(return_queue, main, news, tops)
            if self.check(main, news, tops, tops_available):
                for process in processes:
                    process.terminate()
                logging.info('Domain ' + domain + ': SUCCESS\n')
                return True
            logging.info("Domain " + domain + " failed, waiting.")
            self.print_difference(main, news, tops, tops_available, main_url, news_url, tops_url)
            time.sleep(self.CHECK_PERIOD_SEC)

        for process in processes:
            process.join()
        self.fill_tops_data(return_queue, main, news, tops)

        if (self.check(main, news, tops, tops_available)):
            logging.info('Domain ' + domain + ': SUCCESS\n')
            return True
        logging.info('Domain ' + domain + ': FAILED')
        self.print_difference(main, news, tops, tops_available, main_url, news_url, tops_url)
        return False

    class Parameters(sdk2.Task.Parameters):
        domains = sdk2.parameters.List('Domains', required=True)
        test_id = sdk2.parameters.String('test_id', required=True)
        rank_extra = sdk2.parameters.String('rank_extra', default='')
        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024
        cores = 1
        ram = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        success = True
        for domain in self.Parameters.domains:
            if not self.check_domain(domain):
                success = False
                break
        if not success:
            raise common.errors.TaskFailure('Failed')
