import json
import Queue
import time
import imp
import logging
import datetime

from sandbox.projects.common.search.requester import format_users_queries_line
from sandbox.sandboxsdk.channel import channel
from sandbox.sdk2.service_resources import TaskLogs
from sandbox.sdk2 import ResourceData


def line_requests_list(fname):
    with open(fname) as f:
        return [line.rstrip('\n') for line in f.readlines()]


def json_to_str(js):
    return json.dumps(js, sort_keys=True, indent=2)


STATES_REQ = ["TO_SEND", "SEND", "TO_PROCESS", "OK", "ERROR"]


class Box:

    def __init__(self, task, config, file_path, host, yt_log_table, yt_token, test_flag=False):
        self.time = 0
        self.yt_token = yt_token
        self.yt_log_table = yt_log_table
        self.file_path = file_path
        self.owners = config.get('owners')
        self.name = config.get('name')
        self.host = config.get('host', host)
        self.box = config.get('box')
        self.flags = config.get('flags')
        self.map = config.get('map')
        self.fill_functions_executable(self.map, file_path)
        self.reduce = config.get('reduce')
        self.fill_functions_executable(self.reduce, file_path)
        self.qresp = Queue.Queue()
        box_path = file_path + '/boxes/' + self.box
        self.type = config.get('type', 'search')
        if self.type == 'search':
            self.batch = [
                {
                    "uri": '{host}/search/?{req}&{flags}&json_dump=reqdata.reqid'.format(host=self.host, req=format_users_queries_line(req), flags=self.flags),
                    "id": str(idx),
                } for idx, req in enumerate(line_requests_list(box_path))
            ]
        elif self.type == 'profile':
            self.batch = []
            for idx, req in enumerate(line_requests_list(box_path)):
                req = req.rstrip('\n').split('\t')
                if len(req) > 1:
                    lr = req[1]
                else:
                    lr = 213
                self.batch.append({
                    "uri": '{host}/profile/{pid}?lr={lr}&{flags}&json_dump=reqdata.reqid'.format(host=self.host, pid=req[0], lr=lr, flags=self.flags),
                    "id": str(idx),
                })
        else:
            raise 'Unknown box type {}'.format(self.type)

        print("builded batch:", self.batch)
        self.size = len(self.batch)
        self.responses = [None] * self.size
        self.map_ans = dict()
        self.reduce_ans = dict()
        self.reqids = [""] * self.size
        self.errors = [""] * self.size
        self.states = ["ERROR"] * self.size
        self.flag = True
        self.state = None
        self.test_flag = test_flag
        self.log = ""
        self.task = task

    def print_log(self, msg, flag=3):
        if flag & 1:
            logging.debug(msg)
        if flag & 2 and self.test_flag:
            self.log += '\n' + str(datetime.datetime.now()) + ":" + msg + "\n"

    def _get_response(self):
        resp = self.qresp.get_nowait()
        return resp

    def put_response(self, num, ok, resp):
        self.qresp.put_nowait((num, ok, resp))

    def get_test_flag(self):
        return self.test_flag

    def get_time(self):
        return self.time

    def get_name(self):
        return self.name

    @staticmethod
    def fill_functions_executable(funcs, file_path):
        for func in funcs:
            path = file_path + '/functions/' + func.get('path')
            module = imp.load_source('module.name', path)
            func['executable'] = vars(module)[func.get('name')]

    def process_response(self):
        while not self.qresp.empty():
            num, ok, value = self._get_response()
            if ok:
                self.responses[num] = value
                self.states[num] = "TO_PROCESS"
                self.print_log("response" + str(num) + ":" + json_to_str(value), 2)
            else:
                self.error_in_process(value, num, '')

    def process(self):
        for idx, resp in enumerate(self.responses):
            if self.states[idx] != "TO_PROCESS":
                continue
            if (isinstance(resp, dict)):
                self.reqids[idx] = resp.get("reqdata.reqid", "")
            try:
                for func in self.map:
                    if not func.get('args') is None:
                        result = func.get('executable')(resp, **(func.get('args')))
                    else:
                        result = func.get('executable')(resp)

                    if self.map_ans.get(func.get('output')) is None:
                        self.map_ans[func.get('output')] = []

                    logging.debug('map ('+str(idx)+') = '+str(result))
                    self.map_ans[func['output']].append(result)
                self.states[idx] = 'OK'

            except Exception as er:
                self.error_in_process(er, idx, resp)

    def get_flag(self):
        return self.flag

    def get_state(self):
        return self.state

    def get_reduce_ans(self):
        return self.reduce_ans

    def get_stats(self):
        stat = dict()
        return stat

    def get_for_solomon(self):
        stat = self.get_stats()
        stat.update(self.reduce_ans)
        self.print_log("for solomon:" + str(stat), 2)
        return stat

    def error_in_process(self, error, idx, resp):
        self.errors[idx] = str(error)
        self.states[idx] = "ERROR"
        self.print_log('error {} in {} '.format(error, idx))

    def count_reduce(self):
        for func in self.reduce:
            try:
                if func.get('args') is None:
                    result = func['executable'](self.map_ans)
                else:
                    result = func['executable'](self.map_ans, **(func.get('args')))

                self.reduce_ans[func['sensor_name']] = result
            except Exception as ex:
                self.print_log('Error in count_reduce ' + self.name + ' ' + str(ex))
                return False
        return True

    def save_to_yt(self):
        data = []
        for idx in range(len(self.batch)):
            row = dict()
            row.update(self.batch[idx])
            row["map"] = [{key: val[idx]} if len(val) > idx else {key: None} for key, val in self.map_ans.items()]
            row["reduce"] = self.reduce_ans
            row["state"] = self.states[idx]
            row["response"] = self.responses[idx]
            row["name"] = self.name
            row["error"] = self.errors[idx]
            row["reqid"] = self.reqids[idx]
            data.append(row)
        import yt.wrapper as yt
        yt.config["proxy"]["url"] = "hahn"
        yt.config["token"] = self.yt_token
        yt.write_table(yt.TablePath(self.yt_log_table, append=True), data, format='json')

    def run(self):
        self.time -= time.time()
        self.process_response()
        self.process()
        self.print_log("states = " + str(self.states))
        self.print_log("map_ans = " + str(self.map_ans))
        if "ERROR" in self.states:
            self.state = False
            msg = 'Error! Box ' + self.name + ' finished with error. You can see report here '
            self.save_to_yt()
        else:
            self.state = self.count_reduce()
            self.print_log("reduce_ans = " + str(self.reduce_ans))
            self.save_to_yt()
            if self.state:
                msg = 'OK! Box ' + self.name + ' finish testing successful. You can see report here '
            else:
                msg = 'Error! Box ' + self.name + ' finished with error in reduce function. You can see report here '
        if self.test_flag or not self.state:
            self.print_log("sending email to " + str(self.owners))
            resource = TaskLogs(self.task, "Box_Report", "Reports")
            resource_data = ResourceData(resource)
            resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
            filename = self.name + "-" + str(datetime.datetime.now()).replace(' ', '_') + ".txt"
            resource_data.path.joinpath(filename).write_bytes(str(self.log))
            url = 'https://proxy.sandbox.yandex-team.ru/task/' + str(channel.task.id) + '/Reports/' + filename
            channel.sandbox.send_email(mail_to=self.owners, mail_cc='', mail_subject='NoapacheUpperTest report', mail_body=msg + ' ' + url)
