# -*- coding: utf-8 -*-
import datetime
import logging
import tarfile
from contextlib import closing
from os import path, mkdir

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask


class OcrImageLogFolder(SandboxStringParameter):
    name = 'ocr_image_log'
    description = 'New ocr-api image logs on banach'
    default_value = '//home/cvtest/ocr/logs/ocr_api/image_logs'
    group = 'OCR Api logs'
    multiline = False


class OcrTextLogFolder(SandboxStringParameter):
    name = 'ocr_text_log'
    description = 'New ocr-api text logs on banach'
    default_value = '//home/cvtest/ocr/logs/ocr_api/text_logs'
    group = 'OCR Api logs'
    multiline = False


class OcrMergedLogTableParts(SandboxStringParameter):
    name = 'ocr_merged_log_parts'
    description = 'Ocr-api parts logs'
    default_value = '//home/cvtest/ocr/logs/ocr_api/logs.part'
    group = 'OCR Api logs'
    multiline = False


class OcrMergedLogTable(SandboxStringParameter):
    name = 'ocr_merged_log'
    description = 'Merged ocr-api logs'
    default_value = '//home/cvtest/ocr/logs/ocr_api/logs'
    group = 'OCR Api logs'
    multiline = False


class OcrArchiveLogPath(SandboxStringParameter):
    name = 'ocr_archive_log'
    description = 'Archive ocr-api logs'
    default_value = '//home/cvtest/ocr/logs/ocr_api/archive'
    group = 'OCR Api logs'
    multiline = False


class OlderThenDays(SandboxIntegerParameter):
    name = 'ocr_older_then_days'
    description = 'Days before archiving'
    default_value = 7
    group = 'OCR Api logs'


class DeleteImages(SandboxBoolParameter):
    name = 'ocr_delete_images'
    description = 'Remove images in archive'
    default_value = True
    group = 'OCR Api logs'


class DeleteOcrResult(SandboxBoolParameter):
    name = 'ocr_delete_ocr_result'
    description = 'Remove ocr result in archive'
    default_value = False
    group = 'OCR Api logs'


class OcrApiKeys(SandboxStringParameter):
    name = 'ocr_api_keys'
    description = 'Api keys separated \n'
    default_value = ''
    multiline = True
    group = 'OCR Api logs'


class DateBase(SandboxStringParameter):
    @classmethod
    def cast(cls, value):
        datetime.datetime.strptime(value, '%Y.%m.%d:%H.%M.%S')
        return value


class StartDate(DateBase):
    name = 'start_date'
    description = 'Start date (YYYY.MM.DD:hh.mm.ss)'
    default_value = (datetime.datetime.today() - datetime.timedelta(days=2)).strftime("%Y.%m.%d:%H.%M.%S")
    group = 'OCR Api logs'


class EndDate(DateBase):
    name = 'end_date'
    description = 'End date (YYYY.MM.DD:hh.mm.ss)'
    default_value = datetime.datetime.today().strftime("%Y.%m.%d:%H.%M.%S")
    group = 'OCR Api logs'


class CreateTarArchive(SandboxBoolParameter):
    name = 'ocr_create_tar_archive'
    description = 'Create tar archive'
    default_value = False
    group = 'OCR Api logs'


class OcrLogsTypeParameter(SandboxStringParameter):
    name = 'ocr_logs_run_type'
    description = 'Type of ...'
    required = True
    default_value = 'merge'
    group = 'OCR Api logs'
    choices = [
        ('merge raw logs', 'merge'),
        ('grep logs', 'grep'),
        ('archiving logs', 'archive'),
    ]
    sub_fields = {
        'merge': [OcrTextLogFolder.name, OcrImageLogFolder.name, OcrMergedLogTableParts.name, OcrMergedLogTable.name],
        'grep': [StartDate.name, EndDate.name, OcrMergedLogTable.name, OcrApiKeys.name, CreateTarArchive.name],
        'archive': [OcrMergedLogTableParts.name, OcrMergedLogTable.name, OlderThenDays.name, OcrArchiveLogPath.name, DeleteImages.name, DeleteOcrResult.name]
    }


class YtToken(SandboxStringParameter):
    name = 'yt_token'
    description = 'Vault for token'
    default_value = ''
    group = 'Token for YT'
    multiline = False


class ArcadiaPathToScripts(SandboxSvnUrlParameter):
    name = 'path_to_scripts_in_arcadia'
    description = 'SVN URL Path to scripts in arcadia'
    default_value = 'arcadia:/arc/trunk/arcadia/'
    group = 'Arcadia'


class OcrApiLogs(ArcadiaTask):
    type = 'OCR_API_LOGS'

    LOCAL_SCRIPTS_PATH = 'scripts'
    LOCAL_GREP_PATH = 'grep_result'
    YT_PROXY = 'banach'
    SCRIPT_LOGS = 'scriptlogs.log'
    YT_LOCAL_TMP = 'tmp_yt_data'

    input_parameters = [
        OcrLogsTypeParameter,
        OcrImageLogFolder,
        OcrTextLogFolder,
        OcrMergedLogTable,
        OcrMergedLogTableParts,
        StartDate,
        EndDate,
        OcrApiKeys,
        CreateTarArchive,
        OlderThenDays,
        OcrArchiveLogPath,
        DeleteImages,
        DeleteOcrResult,
        ArcadiaPathToScripts,
        YtToken,
    ]

    execution_space = 10000

    def on_execute(self):
        logging.info('Get arcadia paths with scripts')
        build_system = consts.YMAKE_BUILD_SYSTEM
        build_type = consts.RELEASE_BUILD_TYPE
        def_flags = ''
        results_dir = self.abs_path(self.LOCAL_SCRIPTS_PATH)
        arcadia_src_dir = Arcadia.get_arcadia_src_dir(self.ctx[ArcadiaPathToScripts.name])
        if not arcadia_src_dir:
            raise SandboxTaskFailureError(
                'Cannot get repo for url {0}'.format(self.ctx[ArcadiaPathToScripts.name])
            )
        sdk.do_build(
            build_system, arcadia_src_dir,
            [
                'cv/imageproc/ocr/tools/ocr_api_log_processing/merge_ocr_api_logs_yt',
                'cv/imageproc/ocr/tools/ocr_api_log_processing/grep_ocr_api_logs_yt',
                'cv/imageproc/ocr/tools/ocr_api_log_processing/archive_ocr_api_logs_yt'
            ],
            build_type, clear_build=True,
            results_dir=results_dir, def_flags=def_flags
        )
        logging.info('Check wirking type')
        w_type = self.ctx[OcrLogsTypeParameter.name]
        execCommand = ''
        params = []

        create_tar_archive = False
        save_resource_path = ''

        yt_local_tmp_abs = self.abs_path(self.YT_LOCAL_TMP)
        mkdir(yt_local_tmp_abs)
        ytToken = self.ctx[YtToken.name]
        # for local tests
        if len(ytToken) != 39:
            ytToken = self.get_vault_data(self.owner, ytToken)

        if w_type == 'merge':
            # script.py yt_proxy yt_token img_logs_path text_logs_path part_logs result_logs
            execCommand = 'bin/merge_ocr_api_logs_yt'
            params = [
                self.YT_PROXY,
                ytToken,
                self.ctx[OcrImageLogFolder.name],
                self.ctx[OcrTextLogFolder.name],
                self.ctx[OcrMergedLogTableParts.name],
                self.ctx[OcrMergedLogTable.name],
                yt_local_tmp_abs,
            ]

        elif w_type == 'grep':
            # script.py yt_proxy yt_token logs_path api_keys time_s time_e out_path
            execCommand = 'bin/grep_ocr_api_logs_yt'
            keys = '"' + self.ctx[OcrApiKeys.name].replace('\n', '\t') + '"'
            resource_path = self.abs_path(self.LOCAL_GREP_PATH)
            save_resource_path = resource_path
            if self.ctx[CreateTarArchive.name]:
                create_tar_archive = True
                resource_path = resource_path + '.tar.gz'
            self._create_resource(
                'ocr_logs',
                resource_path,
                resource_types.OCR_LOG_FOLDER
            )
            params = [
                self.YT_PROXY,
                ytToken,
                self.ctx[OcrMergedLogTable.name],
                keys,
                self.ctx[StartDate.name],
                self.ctx[EndDate.name],
                save_resource_path,
                yt_local_tmp_abs
            ]

        elif w_type == 'archive':
            # script.py yt_proxy yt_token logs_part_path logs_path time archive_path delete_imgs delete_ocr_res
            execCommand = 'bin/archive_ocr_api_logs_yt'
            archive_table = path.join(self.ctx[OcrArchiveLogPath.name], 'log_archive_' + datetime.datetime.now().strftime('%Y.%m.%d:%H.%M.%S'))
            params = [
                self.YT_PROXY,
                ytToken,
                self.ctx[OcrMergedLogTableParts.name],
                self.ctx[OcrMergedLogTable.name],
                str(self.ctx[OlderThenDays.name]),
                archive_table,
                str(self.ctx[DeleteImages.name]),
                str(self.ctx[DeleteOcrResult.name]),
                yt_local_tmp_abs,
            ]

        else:
            raise SandboxTaskFailureError('Incorrect working type')

        execCommand = path.join(self.LOCAL_SCRIPTS_PATH, execCommand)
        command = execCommand + ' ' + ' '.join(params)
        logging.info('Run command: %s' % command.replace(ytToken, "***"))

        run_process(command, shell=True, wait=True, log_prefix=self.SCRIPT_LOGS, outputs_to_one_file=True, check=True)
        logging.info('Create archive')
        if create_tar_archive:
            with closing(tarfile.open(save_resource_path + '.tar.gz', 'w:gz', dereference=True)) as tar_file:
                tar_file.add(save_resource_path, arcname="./")
        logging.info('Done')


__Task__ = OcrApiLogs
