# -*- coding: utf-8 -*-
import logging
import shutil
import tarfile
import json
import re
from os import path, listdir
from contextlib import closing
from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel


class OcrArcadiaPath(SandboxSvnUrlParameter):
    name = 'checkout_ocr_arcadia'
    description = 'SVN URL (to arcadia)'
    default_value = 'arcadia:/arc/trunk/arcadia/'
    group = 'Build parameters'


class OcrArcadiaDataPath(SandboxSvnUrlParameter):
    name = 'checkout_ocr_data'
    description = 'SVN URL (to arcadia data)'
    default_value = 'arcadia:/arc/trunk/arcadia_tests_data/images/ocr'
    group = 'Ocr'


class OcrLocalYmake(SandboxStringParameter):
    name = 'ocr_local_ymake'
    description = 'Local ymake'
    default_value = ''
    group = 'Build parameters'
    multiline = True


class OcrConfigPath(SandboxStringParameter):
    name = 'ocr_config_path'
    description = 'Path to ocr config'
    default_value = './'
    group = 'Ocr'
    multiline = False


class OcrDaemonConfigPath(SandboxStringParameter):
    name = 'ocr_daemon_config_path'
    description = 'Path to ocrdaemon config'
    default_value = './'
    group = 'Ocr'
    multiline = False


class BuildOcrRunner(SandboxBoolParameter):
    name = 'build_ocr_runner'
    description = 'Build ocr_runner app'
    default_value = False
    required = True
    group = 'Ocr'


class BuildOcrDaemon(SandboxBoolParameter):
    name = 'build_ocr_daemon'
    description = 'Build ocr server daemon'
    default_value = False
    required = True
    group = 'Ocr'


class BuildOcrYt(SandboxBoolParameter):
    name = 'build_ocr_yt'
    description = 'Build ocr app for yt'
    default_value = False
    required = True
    group = 'Ocr'


class AndroidToolKit(LastReleasedResource):
    name = 'android_toolkit'
    description = 'Toolkit for Android'
    required = True
    group = 'Ocr'


class AndroidStripPath(SandboxStringParameter):
    name = 'android_toolkit_strip'
    description = 'Path to strip in toolkit for Android'
    required = True
    group = 'Ocr'


class AndroidTarget(SandboxStringParameter):
    name = 'android_target_platform'
    description = 'Target platform for android'
    default_value = 'clang38-android-armv7a_neon'
    required = True
    group = 'Ocr'


class IOSToolKit(LastReleasedResource):
    name = 'ios_toolkit'
    description = 'Build toolkit for iOS'
    required = True
    group = 'Ocr'


class IOSTarget(SandboxStringParameter):
    name = 'ios_target_platform'
    description = 'Target platform for iOS'
    default_value = 'clang37-ios-armv7'
    required = True
    group = 'Ocr'


class BuildOcrIOS(SandboxBoolParameter):
    name = 'build_ocr_ios'
    description = 'Build ocr lib for iOS'
    default_value = False
    required = True
    group = 'Ocr'
    sub_fields = {
        'true':
            [IOSToolKit.name, IOSTarget.name],
        'false':
            []
    }


class BuildOcrAndroid(SandboxBoolParameter):
    name = 'build_ocr_android'
    description = 'Build ocr lib for Android'
    default_value = False
    required = True
    group = 'Ocr'
    sub_fields = {
        'true':
            [AndroidToolKit.name, AndroidStripPath.name, AndroidTarget.name],
        'false':
            []
    }


class BuildOcrTestArchive(SandboxBoolParameter):
    name = 'build_ocr_test_archive'
    description = 'Build ocr archive for auto-tests'
    default_value = False
    required = True
    group = 'Ocr'


class BuildOcrData(SandboxBoolParameter):
    name = 'build_ocr_data'
    description = 'Build ocr archive with data'
    default_value = False
    required = True
    group = 'Ocr'
    sub_fields = {
        'true':
            [OcrConfigPath.name],
        'false':
            []
    }


class STTickets(nanny.StartrekTicketIdsParameter):
    required = True
    group = 'Nanny'


class BuildOcrForNanny(SandboxBoolParameter):
    name = 'build_ocr_nanny'
    description = 'Build ocr daemon and data forder to release to Nanny'
    default_value = False
    required = True
    group = 'Ocr'
    sub_fields = {
        'true':
            [OcrConfigPath.name,  OcrDaemonConfigPath.name, STTickets.name],
        'false':
            []
    }


class ServiceForceRelease(SandboxBoolParameter):
    name = 'service_force_release'
    description = 'Force release testing with non-canonical config'
    default_value = False
    required = True
    group = 'Release'


class ServiceTranslate(SandboxBoolParameter):
    name = 'service_translate'
    config_dir = 'translate'
    service_name = 'translate_ocr'
    description = 'Release to translate_ocr'
    default_value = False
    required = True
    group = 'Release'


class ServiceApi(SandboxBoolParameter):
    name = 'service_api'
    config_dir = 'ocr_api'
    service_name = 'api_ocr'
    description = 'Release to api_ocr'
    default_value = False
    required = True
    group = 'Release'


class ServiceSearch(SandboxBoolParameter):
    name = 'service_search'
    config_dir = 'search_app'
    service_name = 'ocr'
    description = 'Release to searh_ocr'
    default_value = False
    required = True
    group = 'Release'


class UseTarGz(SandboxBoolParameter):
    name = 'use_tar_gz'
    description = 'Use gzip compress for result'
    default_value = False
    required = True
    group = 'Ocr'


class OcrBuildTest(nanny.ReleaseToNannyTask, ArcadiaTask):
    type = 'OCR_BUILD_TEST'

    OCR_BIN_TEST_PATH = 'cv/imageproc/ocr/tools/statistic/auto_tests'
    OCR_BIN_OCR_RUNNER_PATH = 'cv/imageproc/ocr/ocr_runner'
    OCR_BIN_OCR_DAEMON_PATH = 'yweb/webdaemons/ocrdaemon'
    OCR_BIN_OCR_YT_PATH = 'cv/imageproc/ocr/tools/mr_tools/mr_ocr'
    OCR_BIN_ANDROID_PATH = 'cv/imageproc/ocr/api/mobile_c/dll'
    OCR_BIN_ANDROID_HEADER_PATHS = ['cv/imageproc/ocr/api/mobile_c/sdk/ocr_lib.h', 'cv/imageproc/ocr/api/mobile_c/sdk_factory/sdk_factory.h']
    OCR_BIN_IOS_PATH = 'cv/imageproc/ocr/api/mobile_c/ios'

    OCR_RESULT_OCR_RUNNER_PATH = 'ocr_runner_output'
    OCR_RESULT_OCR_NANNY_PATH = 'ocr_nanny'
    OCR_RESULT_OCR_DAEMON_PATH = 'ocr_daemon_output'
    OCR_RESULT_OCR_YT_PATH = 'ocr_yt_output'
    OCR_RESULT_ANDROID_PATH = 'ocr_android_output'
    OCR_RESULT_ANDROID_TOOLKIT_PATH = 'ocr_android_toolkit'

    OCR_RESULT_PATH = 'result'
    OCR_RESULT_BIN_PATH = 'result/bin'
    OCR_RESULT_DATA_PATH = 'result/data'

    OCR_BUILD_RESULT_PATH = 'build_tmp'
    OCR_DATA_ALL_PATH = 'data_tmp'
    OCR_DEPENDENCE_FILE = 'ocr_dependence.txt'
    OCR_DATA_PATH = 'ocr_data_result'

    OCR_ALLOW_BIN = ['yweb', 'cv', 'local.ymake']

    OCR_SVN_INFO_RESULT = './svn.info'

    input_parameters = [OcrArcadiaPath, OcrLocalYmake,
                        UseTarGz,
                        BuildOcrRunner,
                        BuildOcrDaemon,
                        BuildOcrYt,
                        BuildOcrIOS, IOSToolKit, IOSTarget,
                        BuildOcrAndroid, AndroidToolKit, AndroidStripPath, AndroidTarget,
                        BuildOcrTestArchive,
                        BuildOcrData,
                        BuildOcrForNanny,
                        ServiceApi, ServiceTranslate, ServiceSearch, ServiceForceRelease,
                        OcrArcadiaDataPath, OcrConfigPath, OcrDaemonConfigPath, STTickets]

    execution_space = 30000

    def on_execute(self):
        logging.info('Start')

        logging.info('Check svn paths')
        (codeSvnInfo, dataSvnInfo) = self.check_all_paths()
        logging.info(json.dumps(codeSvnInfo))
        logging.info(json.dumps(dataSvnInfo))
        logging.info('Check svn paths : Done')

        logging.info('Arcadia export')
        with arcadia_sdk.mount_arc_path(Arcadia.replace(self.ctx[OcrArcadiaPath.name], revision=codeSvnInfo['entry_revision'])) as arcadia_src_dir:
            if not arcadia_src_dir:
                raise SandboxTaskFailureError(
                    'Cannot get repo for url {0}'.format(self.ctx[OcrArcadiaPath.name])
                )
            logging.info('Arcadia export : Done')

            comress_mode = ''
            file_ext = '.tar'
            if self.ctx[UseTarGz.name]:
                comress_mode = 'gz'
                file_ext = '.tar.gz'

            ocr_daemon_bin_folder = ''

            if not self.ctx[ServiceForceRelease.name]:
                ocr_config_dir = self.ctx.get(OcrConfigPath.name).split('/')[-2]
                if self.ctx[ServiceTranslate.name] and ocr_config_dir != ServiceTranslate.config_dir:
                    raise SandboxTaskFailureError(
                        'Config path {} and release service {} do not match'.format(ocr_config_dir, ServiceTranslate.config_dir)
                    )
                if self.ctx[ServiceApi.name] and ocr_config_dir != ServiceApi.config_dir:
                    raise SandboxTaskFailureError(
                        'Config path {} and release service {} do not match'.format(ocr_config_dir, ServiceApi.config_dir)
                    )

            if self.ctx[BuildOcrRunner.name]:
                logging.info('Build ocr_runner')
                ocr_runner_build_path = self.abs_path(self.OCR_RESULT_OCR_RUNNER_PATH) + '_tmp'
                make_folder(ocr_runner_build_path, True)
                self.build_all(arcadia_src_dir, [self.OCR_BIN_OCR_RUNNER_PATH], ocr_runner_build_path, self.ctx[OcrLocalYmake.name])
                logging.info('Create resource')
                resource_result_path = self.abs_path(self.OCR_RESULT_OCR_RUNNER_PATH) + file_ext
                resource_result_id = self._create_resource('ocr_runner_result', resource_result_path, resource_types.OCR_RUNNER_ARCHIVE).id
                with closing(tarfile.open(resource_result_path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(path.join(ocr_runner_build_path, self.OCR_BIN_OCR_RUNNER_PATH), arcname="")
                self.mark_resource_ready(resource_result_id)
                logging.info('Create resource : Done')
                logging.info('Build ocr_runner: Done')

            if self.ctx[BuildOcrDaemon.name] or self.ctx[BuildOcrForNanny.name]:
                logging.info('Build ocr_daemon')
                ocr_daemon_build_path = self.abs_path(self.OCR_RESULT_OCR_DAEMON_PATH) + '_tmp'
                make_folder(ocr_daemon_build_path, True)
                self.build_all(arcadia_src_dir, [self.OCR_BIN_OCR_DAEMON_PATH], ocr_daemon_build_path, self.ctx[OcrLocalYmake.name])
                ocr_daemon_bin_folder = path.join(ocr_daemon_build_path, self.OCR_BIN_OCR_DAEMON_PATH)
                logging.info('Create resource')
                resource_result_path = self.abs_path(self.OCR_RESULT_OCR_DAEMON_PATH) + file_ext
                resource_result_id = self._create_resource('ocr_daemon_result', resource_result_path, resource_types.OCR_RUNNER_ARCHIVE).id
                with closing(tarfile.open(resource_result_path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(ocr_daemon_bin_folder, arcname="")
                self.mark_resource_ready(resource_result_id)
                logging.info('Create resource : Done')
                logging.info('Build ocr_daemon: Done')

            if self.ctx[BuildOcrYt.name]:
                logging.info('Build ocr_yt')
                ocr_yt_build_path = self.abs_path(self.OCR_RESULT_OCR_YT_PATH) + '_tmp'
                make_folder(ocr_yt_build_path, True)
                self.build_all(arcadia_src_dir, [self.OCR_BIN_OCR_YT_PATH], ocr_yt_build_path, self.ctx[OcrLocalYmake.name])
                logging.info('Create resource')
                resource_result_path = self.abs_path(self.OCR_RESULT_OCR_YT_PATH) + file_ext
                resource_result_id = self._create_resource('ocr_yt_result', resource_result_path, resource_types.OCR_RUNNER_ARCHIVE).id
                with closing(tarfile.open(resource_result_path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(path.join(ocr_yt_build_path, self.OCR_BIN_OCR_YT_PATH), arcname="")
                self.mark_resource_ready(resource_result_id)
                logging.info('Create resource : Done')
                logging.info('Build ocr_yt: Done')

            if self.ctx[BuildOcrAndroid.name]:
                logging.info('Build ocr_android')
                ocr_android_build_path = self.abs_path(self.OCR_RESULT_ANDROID_PATH) + '_tmp'
                make_folder(ocr_android_build_path, True)
                self.build_all(arcadia_src_dir, [self.OCR_BIN_ANDROID_PATH], ocr_android_build_path, self.ctx[OcrLocalYmake.name], self.ctx[AndroidTarget.name])
                # strip
                ocr_android_build_path_lib = path.join(ocr_android_build_path, 'lib')
                ocr_android_build_path_build_result = path.join(ocr_android_build_path, self.OCR_BIN_ANDROID_PATH)
                for elem in listdir(ocr_android_build_path_build_result):
                    if not path.exists(path.join(ocr_android_build_path_lib, elem)):
                        shutil.copy(path.join(ocr_android_build_path_build_result, elem), ocr_android_build_path_lib)
                logging.info('Strip ocr_android')
                toolKit = self.sync_resource(self.ctx[AndroidToolKit.name])
                with closing(tarfile.open(toolKit, 'r:*', dereference=True)) as tar_file:
                    tar_file.extractall(self.abs_path(self.OCR_RESULT_ANDROID_TOOLKIT_PATH))
                strip_tool = path.join(self.abs_path(self.OCR_RESULT_ANDROID_TOOLKIT_PATH), self.ctx[AndroidStripPath.name])
                for elem in listdir(ocr_android_build_path_lib):
                    command = strip_tool + ' --strip-all ' + path.join(ocr_android_build_path_lib, elem)
                    logging.info('strip: ' + command)
                    proc = run_process(command, shell=True, wait=True, outputs_to_one_file=True, check=False)
                    logging.info('Strip exit code: ' + str(proc.returncode))
                for header_file in self.OCR_BIN_ANDROID_HEADER_PATHS:
                    shutil.copy(path.join(self.abs_path(arcadia_src_dir), header_file), ocr_android_build_path_lib)
                logging.info('Strip ocr_android:Done')
                logging.info('Create resource')
                resource_result_path = self.abs_path(self.OCR_RESULT_ANDROID_PATH) + file_ext
                resource_result_id = self._create_resource('ocr_android_result', resource_result_path, resource_types.OCR_RUNNER_ARCHIVE).id
                with closing(tarfile.open(resource_result_path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(ocr_android_build_path_lib, arcname="")
                self.mark_resource_ready(resource_result_id)
                logging.info('Create resource : Done')
                logging.info('Build ocr_android: Done')

            if self.ctx[BuildOcrIOS.name]:
                logging.info('Build ocr_ios: Not implemented =(')

            if self.ctx[BuildOcrTestArchive.name]:
                logging.info('Build ocr_archive for tests')
                make_folder(self.OCR_RESULT_PATH, True)
                make_folder(self.OCR_RESULT_BIN_PATH, True)
                with open(path.join(self.OCR_RESULT_PATH, self.OCR_SVN_INFO_RESULT), "w") as fOut:
                    fOut.write(json.dumps({"CodeInfo": codeSvnInfo, "DataInfo": dataSvnInfo}))
                self.build_all(arcadia_src_dir, [self.OCR_BIN_TEST_PATH], self.OCR_RESULT_BIN_PATH, self.ctx[OcrLocalYmake.name])
                for elem in listdir(self.OCR_RESULT_BIN_PATH):
                    if not(elem in self.OCR_ALLOW_BIN):
                        remove_path(path.join(self.OCR_RESULT_BIN_PATH, elem))
                Arcadia.export(self.ctx[OcrArcadiaDataPath.name], self.abs_path(self.OCR_RESULT_DATA_PATH), revision=dataSvnInfo['entry_revision'])
                resource_result = channel.sandbox.get_resource(self.ctx['out_ocr_result_id'])
                with closing(tarfile.open(resource_result.path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(self.abs_path(self.OCR_RESULT_PATH), arcname="")
                self.mark_resource_ready(resource_result)
                logging.info('Build ocr_archive for tests: Done')

            if self.ctx[BuildOcrData.name] or self.ctx[BuildOcrForNanny.name]:
                logging.info('Build ocr_data')
                make_folder(self.OCR_BUILD_RESULT_PATH, True)
                make_folder(self.OCR_DATA_ALL_PATH, True)
                make_folder(self.OCR_DATA_PATH, True)
                self.build_all(arcadia_src_dir, [self.OCR_BIN_OCR_RUNNER_PATH], self.OCR_BUILD_RESULT_PATH, self.ctx[OcrLocalYmake.name])
                Arcadia.export(self.ctx[OcrArcadiaDataPath.name], self.abs_path(self.OCR_DATA_ALL_PATH), revision=dataSvnInfo['entry_revision'])
                command = (
                    path.join(self.abs_path(self.OCR_BUILD_RESULT_PATH), self.OCR_BIN_OCR_RUNNER_PATH + '/ocr_runner') +
                    ' --load_config_only --data_dir ' +
                    path.join(self.abs_path(self.OCR_DATA_ALL_PATH), self.ctx.get(OcrConfigPath.name))
                )
                command = 'strace -xf -eopen -o ' + self.abs_path(self.OCR_DEPENDENCE_FILE) + ' ' + command
                proc = run_process(command, shell=True, wait=True, outputs_to_one_file=True, check=False)
                dependencePaths = set()
                prefix = self.abs_path(self.OCR_DATA_ALL_PATH)
                with open(self.abs_path(self.OCR_DEPENDENCE_FILE), "r") as fPaths:
                    for line in fPaths:
                        match = re.search(r'open\("(.*)", .*\)', line)
                        if match is None:
                            continue
                        path_dep = self.abs_path(match.group(1))
                        if path_dep[:len(prefix)] == prefix:
                            file_path = path_dep[len(prefix):]
                            if file_path[0] == '/':
                                file_path = file_path[1:]
                            dependencePaths.add(path.normpath(file_path))
                self.cp_parents(self.abs_path(self.OCR_DATA_ALL_PATH), self.abs_path(self.OCR_DATA_PATH), dependencePaths)
                make_folder(path.join(self.OCR_DATA_PATH, 'ocr_config'), True)
                shutil.copy(path.join(self.abs_path(self.OCR_DATA_ALL_PATH), self.ctx.get(OcrConfigPath.name)), path.join(self.OCR_DATA_PATH, 'ocr_config/ocr.cfg'))
                resource_result_path = self.abs_path(self.OCR_DATA_PATH) + file_ext
                resource_result_id = self._create_resource('ocr_data_result', resource_result_path, resource_types.OCR_RUNNER_ARCHIVE).id
                with closing(tarfile.open(resource_result_path, 'w:' + comress_mode, dereference=True)) as tar_file:
                    tar_file.add(self.abs_path(self.OCR_DATA_PATH), arcname="")
                self.mark_resource_ready(resource_result_id)
                logging.info('Build ocr_data: Done')

            if self.ctx[BuildOcrForNanny.name]:
                logging.info('Build ocr for Nanny')

                # create folder with resource
                make_folder(self.OCR_RESULT_OCR_NANNY_PATH, True)
                filename = self.OCR_RESULT_OCR_NANNY_PATH
                resource_path = self.abs_path(filename)
                resource_name = 'Ocr for Nanny: %s' % (self.descr)
                resource_nanny_id = self._create_resource(resource_name, resource_path, resource_types.OCR_NANNY_FOLDER).id

                # copy bin
                daemon_bin = path.join(ocr_daemon_bin_folder, resource_types.OCR_NANNY_FOLDER.daemon_path)
                if ocr_daemon_bin_folder == '' or not path.exists(daemon_bin):
                    raise SandboxTaskFailureError('OcrDaemon bin does not exist {}'.format(daemon_bin))
                shutil.copy(daemon_bin, path.join(resource_path, resource_types.OCR_NANNY_FOLDER.daemon_path))

                # copy data
                if not path.exists(self.OCR_DATA_PATH):
                    raise SandboxTaskFailureError('Ocr data bin does not exist {}'.format(self.OCR_DATA_PATH))
                copy_path(self.OCR_DATA_PATH, path.join(resource_path, resource_types.OCR_NANNY_FOLDER.data_path))

                # copy configs
                configs_path = path.dirname(resource_types.OCR_NANNY_FOLDER.daemon_config_path)
                make_folder(path.join(resource_path, configs_path), True)
                shutil.copy(path.join(self.abs_path(self.OCR_DATA_ALL_PATH), self.ctx.get(OcrConfigPath.name)), path.join(resource_path, configs_path))
                shutil.copy(path.join(self.abs_path(self.OCR_DATA_ALL_PATH), self.ctx.get(OcrDaemonConfigPath.name)), path.join(resource_path, resource_types.OCR_NANNY_FOLDER.daemon_config_path))

                self.mark_resource_ready(resource_nanny_id)
                self.ctx['ocr_nanny_result_id'] = resource_nanny_id
                logging.info('Build ocr for Nanny: Done')

    def build_all(self, arcadiaPath, targetPaths, resultPath, localYmake=None, target_platform=None):
        if localYmake != '' and localYmake is not None:
            with open(path.join(resultPath, "local.ymake"), "w") as fYmake:
                fYmake.write(localYmake)
        build_system = consts.YMAKE_BUILD_SYSTEM
        build_type = consts.RELEASE_BUILD_TYPE
        def_flags = ''
        results_dir = self.abs_path(resultPath)
        sdk.do_build(
            build_system, arcadiaPath, targetPaths, build_type, clear_build=True,
            results_dir=results_dir, def_flags=def_flags, target_platform=target_platform
        )

    def check_all_paths(self):
        for path_to_check in [self.OCR_BIN_TEST_PATH, self.OCR_BIN_OCR_RUNNER_PATH, self.OCR_BIN_OCR_DAEMON_PATH, self.OCR_BIN_OCR_YT_PATH, self.OCR_BIN_ANDROID_PATH, self.OCR_BIN_IOS_PATH]:
            url = Arcadia.append(self.ctx.get(OcrArcadiaPath.name), path_to_check)
            if not Arcadia.check(url):
                raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(url))
        codeSvnInfo = Arcadia.info(self.ctx[OcrArcadiaPath.name])
        svn_data = ''
        dataSvnInfo = -1
        if self.ctx[BuildOcrTestArchive.name] or self.ctx[BuildOcrData.name] or self.ctx[BuildOcrForNanny.name]:
            svn_data = self.ctx.get(OcrArcadiaDataPath.name)
            if not Arcadia.check(svn_data):
                raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(svn_data))
            dataSvnInfo = Arcadia.info(self.ctx[OcrArcadiaDataPath.name])
        if self.ctx[BuildOcrData.name] or self.ctx[BuildOcrForNanny.name]:
            url = Arcadia.append(svn_data, self.ctx.get(OcrConfigPath.name))
            if not Arcadia.check(url):
                raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(url))
        return (codeSvnInfo, dataSvnInfo)

    def cp_parents(self, source_dir, target_dir, files):
        for file in files:
            if path.isfile(path.join(source_dir, file)):
                dest_file = path.join(target_dir, file)
                make_folder(path.dirname(dest_file), False)
                shutil.copy(path.join(source_dir, file), dest_file)

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        if self.ctx[BuildOcrTestArchive.name]:
            file_ext = '.tar'
            if self.ctx[UseTarGz.name]:
                file_ext = '.tar.gz'
            filename = self.OCR_RESULT_OCR_RUNNER_PATH + file_ext
            resource_path = self.abs_path(filename)
            resource_name = '%s (%s)' % (self.descr, filename)
            self.ctx['out_ocr_result_id'] = self._create_resource(
                resource_name,
                resource_path,
                resource_types.OCR_RUNNER_ARCHIVE).id

    def on_release(self, additional_parameters):
        assert 'ocr_nanny_result_id' in self.ctx and self.ctx['ocr_nanny_result_id'] != 0
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('OCR', 'ocr_nanny_token')
        )
        release_services = []

        if self.ctx[ServiceTranslate.name]:
            release_services.append(ServiceTranslate.service_name)

        if self.ctx[ServiceApi.name]:
            release_services.append(ServiceApi.service_name)

        if self.ctx[ServiceSearch.name]:
            release_services.append(ServiceSearch.service_name)

        release_services_full = []

        for rs in release_services:
            if additional_parameters.get('release_status').lower() == 'testing':
                release_services_full.append('testing_' + rs + '_sas')
            if additional_parameters.get('release_status').lower() == 'stable':
                prefix_prod = 'produnction_'

                if rs == ServiceSearch.service_name:
                    prefix_prod = 'production_'

                release_services_full.append(prefix_prod + rs + '_sas')
                release_services_full.append(prefix_prod + rs + '_vla')
                release_services_full.append(prefix_prod + rs + '_man')

        logging.info('Release to: {}'.format(', '.join(release_services_full)))

        for service_id in release_services_full:
            nanny_client.update_service_sandbox_file(
                service_id=service_id,
                task_type=self.type,
                task_id=str(self.id),
                deploy=True,
                skip_not_existing_resources=True,
                recipe='default', prepare_recipe='default'
            )
        ArcadiaTask.on_release(self, additional_parameters)


__Task__ = OcrBuildTest
