# -*- coding: utf-8 -*-

from sandbox import sdk2
import json
import logging
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch.CreateAddrsBeta import CreateAddrsBeta
from sandbox.projects.geosearch.AddrsMapkitCompareTest import AddrsMapkitCompareTest
from sandbox.projects.OrgCollections.resources import OrgCollectionsIndex


class AcceptanceOrgCollectionsIndex(sdk2.Task):
    '''Acceptance task for OrgCollectionsIndex'''

    class Parameters(sdk2.Parameters):
        test_index = sdk2.parameters.Resource(
            'Collections index to accept',
            resource_type=OrgCollectionsIndex,
            required=True
        )

    def _make_beta_url(self, task_id):
        task = sdk2.Task[task_id]
        json_sources = task.Context.experimental_sources
        if json_sources:
            beta_sources = json.loads(json_sources)
            url_tpl = 'http://addrs-testing.search.yandex.net/search/stable/yandsearch?{experimental_sources}'
            experimental_sources = ['&source=%s' % source for source in beta_sources]
            experimental_sources_string = ''.join(experimental_sources)
            return url_tpl.format(experimental_sources=experimental_sources_string)
        else:
            message = "Can't get experimental sources from beta, task id: {}".format(task_id)
            raise SandboxTaskFailureError(message)

    def on_execute(self):
        with self.memoize_stage.prepare_betas:
            logging.info('Preparing betas...')
            create_beta_class = sdk2.Task[CreateAddrsBeta.type]
            create_ref_collections = create_beta_class(create_beta_class.current,
                                                       branch=str(self.id),
                                                       tag='reference',
                                                       resources=[],
                                                       beta_type_choices='collections',
                                                       launch_type_choices='DB',
                                                       owner=self.owner,
                                                       create_sub_task=True).enqueue()
            create_test_collections = create_beta_class(create_beta_class.current,
                                                        branch=str(self.id),
                                                        tag='test',
                                                        resources=[self.Parameters.test_index.id],
                                                        beta_type_choices='collections',
                                                        launch_type_choices='DB',
                                                        owner=self.owner,
                                                        create_sub_task=True).enqueue()
            self.Context.create_ref_collections_id = create_ref_collections.id
            self.Context.create_test_collections_id = create_test_collections.id
            raise sdk2.WaitTask([self.Context.create_ref_collections_id,
                                 self.Context.create_test_collections_id],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)

        with self.memoize_stage.acceptance:
            logging.info('Starting acceptance step...')
            reference_url = self._make_beta_url(self.Context.create_ref_collections_id)
            logging.info('Reference url: {}'.format(reference_url))
            test_url = self._make_beta_url(self.Context.create_test_collections_id)
            logging.info('Test url: {}'.format(test_url))

            acceptance_task_class = sdk2.Task[AddrsMapkitCompareTest.type]
            acceptance_task = acceptance_task_class(acceptance_task_class.current,
                                                    reference_stand_url=reference_url,
                                                    test_stand_url=test_url,
                                                    tests_tag='collections_search',
                                                    launch_type_choices='DB',
                                                    kill_timeout=self.Parameters.kill_timeout,
                                                    owner=self.owner,
                                                    create_sub_task=True).enqueue()
            self.Context.acceptance_task_id = acceptance_task.id
            raise sdk2.WaitTask([self.Context.acceptance_task_id],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)

        acceptance_task = sdk2.Task[self.Context.acceptance_task_id]
        if acceptance_task.status != ctt.Status.SUCCESS:
            raise SandboxTaskFailureError('Acceptance task failed')
        if acceptance_task.Context.diff:
            raise SandboxTaskFailureError('Acceptance failed, diff is not empty')

        logging.info('Done')
