import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects import resource_types
from sandbox.projects.common.geosearch.utils import unpack_files, unpack_file
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.geobase.Geodata4BinStable.resource import GEODATA4BIN_STABLE
from sandbox.projects.geosearch.snippets.AddrsSnippetsTaskManager import AddrsSnippetsTaskManager
from sandbox.projects.geosearch.tools.database_notifications import who_is_on_duty
from sandbox.projects.OrgCollections.resources import OrgCollectionsIndexerExecutable, OrgCollectionsIndex
from sandbox.projects.OrgCollections.accept_index import AcceptanceOrgCollectionsIndex

from sandbox import common
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


# Resources
class MakeExpertCollectionsExecutable(sdk2.Resource):
    """
        YT-tool that processes the experts' collections and saves it to file and YT-table
    """
    executable = True
    releasable = True
    releasers = resource_releasers.geosearch_releasers


# util functions TODO: move to common place
def get_unpacked_source(parameter, target_dir):
    arch_path = str(sdk2.ResourceData(parameter).path)
    unpack_dir = common.fs.make_folder(target_dir)
    unpack_file(arch_path, unpack_dir)
    unpack_files(unpack_dir)
    return unpack_dir


# Main task
class OrgCollectionsIndexer(ReleaseToNannyTask2, sdk2.Task):
    '''Task for building index for orgs collections search'''

    class Parameters(sdk2.Parameters):
        need_acceptance = sdk2.parameters.Bool(
            'Need to launch acceptance task',
            default=True
        )

        expert_collections_threshold = sdk2.parameters.Float(
            'Threshold of expert collections amount in range [0.0, 1.0]',
            default=0.9,
            required=True
        )

        make_expert_collections_executable = sdk2.parameters.Resource(
            'make_expert_collections executable',
            resource_type=MakeExpertCollectionsExecutable,
            required=True
        )

        yt_dir = sdk2.parameters.String(
            'Resulting YT-directory with collections export',
            default='//home/geosearch/collections/production',
            required=True)

        experts_table = sdk2.parameters.String(
            'YT-Path to the table with collections from experts',
            default='//home/geosearch/collections/production/expert_collections',
            required=True)

        geobase_snapshot = sdk2.parameters.Resource(
            'Geobase 4 snapshot',
            resource_type=GEODATA4BIN_STABLE,
            required=True
        )

        static_factors_downloader = sdk2.parameters.Resource(
            'static_factors_downloder executable',
            resource_type=resource_types.GEOSEARCH_STATIC_DOWNLOADER_EXECUTABLE,
            required=True,
        )

        indexer = sdk2.parameters.Resource(
            'indexer',
            resource_type=OrgCollectionsIndexerExecutable,
            required=True,
        )

        with sdk2.parameters.String('Environment', required=True, multiline=True) as environment:
            environment.values['stable'] = environment.Value('stable')
            environment.values['testing'] = environment.Value('testing')

        need_upload_snippets = sdk2.parameters.Bool(
            'Upload snippets to SaaS',
            default=False
        )

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('saaspy'),
        ]

    def execute_cmd(self, executable_parameter, cmd_params):
        executable = str(sdk2.ResourceData(executable_parameter).path)
        logging.info('Run command: {}'.format(' '.join(str(v) for v in ([executable] + cmd_params))))
        with sdk2.helpers.ProcessLog(self, logger='cmd_executor') as l:
            sp.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

    def build_index(self):
        import yt.wrapper as yt

        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        yt.config["proxy"]["url"] = 'hahn.yt.yandex.net'
        yt.config["token"] = yt_token

        logging.info('Creating index resource...')
        index_resource = OrgCollectionsIndex(self, 'Org collections index', 'index')
        index_res_data = sdk2.ResourceData(index_resource)
        index_res_data.path.mkdir(0o755, parents=True, exist_ok=True)

        logging.info('Unzipping geobase...')
        get_unpacked_source(self.Parameters.geobase_snapshot, str(index_res_data.path))

        expert_collections_filename = 'expert_collections.json'
        expert_dir = self.Parameters.yt_dir + "/expert"

        logging.info('Processing collections from experts...')
        self.execute_cmd(
            self.Parameters.make_expert_collections_executable,
            [
                '--experts_table', self.Parameters.experts_table,
                '--out_dir', expert_dir,
                '--out_file', expert_collections_filename,
                '--yt_proxy', 'hahn.yt.yandex.net',
                '--collections_threshold', str(self.Parameters.expert_collections_threshold),
            ]
        )

        logging.info('Downloading static factors...')
        yt_tokenfile = 'yt_token.txt'
        with open(yt_tokenfile, 'w') as output:
            output.write(yt_token)

        expert_static_factors = expert_dir + "/static_factors"

        all_static_factors = yt.create_temp_table()
        yt.concatenate([expert_static_factors], all_static_factors)

        self.execute_cmd(
            self.Parameters.static_factors_downloader,
            [
                '-i', all_static_factors,
                '-o', str(index_res_data.path.joinpath('static_factors.mms')),
                '-k', 'Oid',
                '-s', 'hahn.yt.yandex.net',
                '-t', yt_tokenfile,
            ]
        )

        logging.info('Indexing...')
        self.execute_cmd(
            self.Parameters.indexer,
            [
                '-c', expert_collections_filename,
                '-o', str(index_res_data.path),
            ]
        )

        index_res_data.ready()
        self.Context.index_resource_id = index_resource.id
        logging.info('Index was built')

    def launch_acceptance(self):
        logging.info('Starting acceptance...')
        acceptance_task_class = sdk2.Task[AcceptanceOrgCollectionsIndex.type]
        acceptance_task = acceptance_task_class(acceptance_task_class.current,
                                                test_index=sdk2.Resource[self.Context.index_resource_id],
                                                owner=self.owner,
                                                kill_timeout=self.Parameters.kill_timeout,
                                                create_sub_task=True).enqueue()
        self.Context.acceptance_task_id = acceptance_task.id
        raise sdk2.WaitTask([self.Context.acceptance_task_id],
                            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                            wait_all=True)

    def check_subtask_status(self, task_id, error_msg):
        task = sdk2.Task[task_id]
        if task.status != ctt.Status.SUCCESS:
            raise SandboxTaskFailureError(error_msg)

    def notify_on_duty(self, event):
        try:
            on_duty = who_is_on_duty()
            message = ('Collections index build {event}\n'
                       'Details can be found in '
                       'https://sandbox.yandex-team.ru/task/{task}/view').format(task=self.id, event=event)
            logging.info('Email notification on {event} will be sent to {on_duty}'.format(event=event, on_duty=on_duty))
            self.server.notification(body=message,
                                     recipients=on_duty,
                                     transport=ctn.Transport.EMAIL)
        except Exception:
            logging.exception('Failed to send email notifications')

    def on_failure(self, prev_status):
        sdk2.Task.on_failure(self, prev_status)
        self.notify_on_duty('failed')

    def on_break(self, prev_status, status):
        sdk2.Task.on_break(self, prev_status, status)
        self.notify_on_duty('broken')

    def on_execute(self):
        with self.memoize_stage.build_index:
            self.build_index()

        with self.memoize_stage.acceptance:
            if self.Parameters.need_acceptance:
                self.launch_acceptance()

        if self.Parameters.need_acceptance:
            self.check_subtask_status(self.Context.acceptance_task_id,
                                       "Index acceptance failed")

        with self.memoize_stage.snippets_uploading:
            if self.Parameters.need_upload_snippets:
                self.upload_snippets()

        if self.Parameters.need_upload_snippets:
            self.check_subtask_status(self.Context.task_manager_id,
                                       "Snippets uploading failed")

        logging.info('Done')

    def upload_snippets(self):
        logging.info('Starting uploading snippets...')
        collections_config = \
            'arcadia:/arc/trunk/arcadia/search/geo/tools/task_manager/configs/collections_sync.json'
        task_class = sdk2.Task[AddrsSnippetsTaskManager.type]
        task_manager = task_class(self,
                                  task_manager_config=collections_config,
                                  mail_list=[],
                                  owner=self.owner,
                                  notifications=self.Parameters.notifications,
                                  create_subtask=True)
        task_manager.enqueue()
        self.Context.task_manager_id = task_manager.id
        raise sdk2.WaitTask([self.Context.task_manager_id],
                            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                            wait_all=True)
