import logging
import os
import json
from datetime import datetime

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects import resource_types
from sandbox.projects.common.solomon import push_to_solomon_v2

from sandbox.sandboxsdk import environments


# Resource types
class OrgEventsFilterExecutable(sdk2.Resource):
    """
        Utility that removes snippets with deleted events and updates merged table
    """
    executable = True
    releasable = True
    releasers = resource_releasers.geosearch_releasers


class OrgEventsFilter(sdk2.Task):
    '''Task that filters removed events and updates snippets'''

    class Parameters(sdk2.Parameters):
        events_filter_executable = sdk2.parameters.Resource(
            'events_filter executable',
            resource_type=OrgEventsFilterExecutable,
            required=True
        )

        snippets_builder_executable = sdk2.parameters.Resource(
            'snippets_builder executable',
            resource_type=resource_types.SOCIAL_EVENTS_SNIPPET_BUILDER_EXECUTABLE,
            required=True
        )

        bad_links_tables = sdk2.parameters.List('Tables with broken links', required=True)

        update_tables = sdk2.parameters.List('Tables for update (needed to trigger Aurora)', required=True)

        actual_profiles_table = sdk2.parameters.String('Table with actual profiles', required=True)

        modified_profiles_table = sdk2.parameters.String('Table where to store modified profiles', required=True)

        merged_table = sdk2.parameters.String('Table that stores all profiles with posts')

        dump_to_indexerproxy_executable = sdk2.parameters.Resource(
            'dump_to_indexerproxy executable',
            resource_type=resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE,
        )

        with sdk2.parameters.String('Environment', required=True, multiline=True) as environment:
            environment.values['stable'] = environment.Value('stable')
            environment.values['testing'] = environment.Value('testing')

        send_stats = sdk2.parameters.Bool('Send stats to Solomon', default=True)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('saaspy'),
        ]

    def execute_cmd(self, executable_parameter, cmd_params, log_name):
        executable = str(sdk2.ResourceData(executable_parameter).path)
        logging.info('Run command: {}'.format(' '.join([executable] + cmd_params)))
        with sdk2.helpers.ProcessLog(self, logger=log_name) as l:
            sp.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

    def create_sensors(self, data):
        sensors = []
        timestamp = int(datetime.now().strftime('%s'))
        for origin in data:
            for k, v in data[origin].items():
                sensors.append({
                    'labels': {
                        'posts_origin': origin,
                        'sensor': k,
                    },
                    'ts': timestamp,
                    'value': float(v),
                })
        return sensors

    def create_sensors_from_file(self, filename):
        return self.create_sensors(json.load(open(filename)))

    def on_execute(self):
        import yt.wrapper as yt
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')
        solomon_token = sdk2.Vault.data(self.owner, 'solomon-token')
        yt.config['token'] = yt_token
        yt.config['proxy']['url'] = 'hahn.yt.yandex.net'

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        solomon_common_labels = {
            'project': 'geosearch_social_links',
            'cluster': 'org_social_links',
            'service': self.Parameters.environment,
        }

        if yt.get_attribute(self.Parameters.actual_profiles_table, 'lock_count') != 0 or \
           yt.get_attribute(self.Parameters.merged_table, 'lock_count') != 0:
            logging.info('Input tables are locked, exiting')
            return

        logging.info('Filtering snippets...')
        self.execute_cmd(
            self.Parameters.events_filter_executable,
            self.Parameters.bad_links_tables + [
                self.Parameters.actual_profiles_table,
                self.Parameters.modified_profiles_table,
                '--proxy', 'hahn.yt.yandex.net',
                '--out_stats', 'filter_stats.json',
            ] + (['--merged_table', self.Parameters.merged_table] if self.Parameters.merged_table else []),
            'events_filter'
        )

        if self.Parameters.send_stats:
            solomon_common_labels['host'] = 'events_filter'
            push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors_from_file('filter_stats.json'))

        modified_snippets_table = self.Parameters.modified_profiles_table + "_snippets"
        logging.info('Building snippets...')
        self.execute_cmd(
            self.Parameters.snippets_builder_executable,
            [
                self.Parameters.modified_profiles_table,
                modified_snippets_table
            ],
            'snippets_builder'
        )

        for table in self.Parameters.update_tables:
            yt.set_attribute(table, 'update_flag', 'update')

        logging.info('Uploading snippets to saas-kv...')
        if self.Parameters.dump_to_indexerproxy_executable:
            dumper_log_path = modified_snippets_table + '_upload.err'
            self.execute_cmd(
                self.Parameters.dump_to_indexerproxy_executable,
                [
                    '--mode', 'mr',
                    '-k', sdk2.Vault.data('GEOMETA-SEARCH', 'saas_org_events_snippets_key'),
                    '-h', 'saas-indexerproxy-maps-kv.yandex.net',
                    '--mr-server', 'hahn.yt.yandex.net',
                    '--mr-input', modified_snippets_table,
                    '--mr-processor', 'cards_geo',
                    '--action', 'atModify',
                    '--mr-jobs', '1000',
                    '--mr-output', dumper_log_path,
                    '--prefix', '1',
                    '--mr-processor-env', 'events/1.x'
                ],
                'dump_to_indexerproxy'
            )
            if self.Parameters.send_stats:
                solomon_common_labels['host'] = 'dumper_filter'
                if yt.exists(dumper_log_path):
                    push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors({
                        'errors': {
                            'dumper_errors': yt.table_commands.row_count(dumper_log_path)
                        }
                    }))
        else:
            logging.info('No dump_to_indexerproxy_executable were provided')

        logging.info('Done')
