import logging
import os
import json
from datetime import datetime

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects import resource_types
from sandbox.projects.common.utils import create_misc_resource_and_dir
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects.common.solomon import push_to_solomon_v2

'''Resources'''


class ORG_EVENTS_UPDATER_RESULT(resource_types.AbstractResource):
    """
        Output resourse for ORG_EVENTS_UPDATER task
    """
    releasers = resource_releasers.geosearch_releasers
    releasable = True


''' Input parameters'''


class Environment_(sdk_parameters.SandboxStringParameter):
    name = 'environment_'
    description = 'Environment_'
    choices = [
        ('testing', 'testing'),
        ('stable', 'stable'),
    ]
    default_value = 'testing'


class SaasDumper(sdk_parameters.ResourceSelector):
    name = 'saas_dumper'
    description = 'SaaS dumper: '
    resource_type = resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE


class AuroraMerger(sdk_parameters.ResourceSelector):
    name = 'aurora_merger'
    description = 'Aurora merger: '
    resource_type = resource_types.SOCIAL_PROFILES_MERGER_FROM_AURORA_EXECUTABLE


class FactExtractor(sdk_parameters.ResourceSelector):
    name = 'fact_extractor'
    description = 'Fact extractor: '
    resource_type = resource_types.SOCIAL_PROFILES_FACTS_EXTRACTOR_EXECUTABLE


class YtBackuper(sdk_parameters.ResourceSelector):
    name = 'yt_backuper'
    description = 'Yt-tables displace backuper'
    resource_type = resource_types.YT_BACKUPER_EXECUTABLE


class ActualProfilesBuilder(sdk_parameters.ResourceSelector):
    name = 'actual_profiles_builder'
    description = 'Actual profiles builder: '
    resource_type = resource_types.SOCIAL_ACTUAL_PROFILES_BUILDER_EXECUTABLE


class SnippetsBuilder(sdk_parameters.ResourceSelector):
    name = 'snippets_builder'
    description = 'Snippets builder: '
    resource_type = resource_types.SOCIAL_EVENTS_SNIPPET_BUILDER_EXECUTABLE


class ExportPeriod(sdk_parameters.SandboxStringParameter):
    name = 'export_period'
    description = 'Export period (in seconds. 0 means infinity): '
    default_value = '0'


class MaxPostAge(sdk_parameters.SandboxStringParameter):
    name = 'max_post_age'
    description = 'Maximal post age (in seconds. 0 means infinity): '
    default_value = str(30 * 24 * 60 * 60)


class ImagesMdsUploader(sdk_parameters.ResourceSelector):
    name = 'images_mds_uploader'
    description = 'Uploader images to mds'
    resource_type = resource_types.SOCIAL_IMAGES_MDS_UPLOADER


class SendStats(sdk_parameters.SandboxBoolParameter):
    name = 'send_stats'
    description = 'Send stats to Solomon'
    default_value = True


class Sources(sdk_parameters.ListRepeater, sdk_parameters.SandboxStringParameter):
    name = 'source_tables'
    description = 'Tables with input to Aurora'


class SolomonToken(sdk_parameters.SandboxStringParameter):
    name = 'solomon-token'
    description = 'Key to owner`s solomon-token in Vault'
    default_value = 'solomon-token'


class Task(sdk_task.SandboxTask):

    '''
        Task that merges updates from aurora, extracts events and uploads snippets to saas-kv
    '''

    type = 'ORG_EVENTS_UPDATER'

    input_parameters = [
        AuroraMerger,
        FactExtractor,
        ImagesMdsUploader,
        YtBackuper,
        ActualProfilesBuilder,
        SnippetsBuilder,
        Environment_,
        SaasDumper,
        ExportPeriod,
        MaxPostAge,
        SendStats,
        Sources,
        SolomonToken
    ]

    environment = (
        sdk_environments.PipEnvironment('yandex-yt'),
        sdk_environments.PipEnvironment('requests'),
        sdk_environments.PipEnvironment('saaspy'),
    )

    profiles_dir = '//home/geosearch/social_networks/profiles/'
    full_profiles_dir = profiles_dir + 'full/'

    def create_sensors(self, data):
        sensors = []
        timestamp = int(datetime.now().strftime('%s'))
        for origin in data:
            for k, v in data[origin].items():
                sensors.append({
                    'labels': {
                        'posts_origin': origin,
                        'sensor': k,
                    },
                    'ts': timestamp,
                    'value': float(v),
                })
        return sensors

    def create_sensors_from_file(self, filename):
        return self.create_sensors(json.load(open(filename)))

    def on_execute(self):
        import yt.wrapper as yt

        yt_token = self.get_vault_data(self.owner, 'yt-token')
        solomon_token = self.get_vault_data(self.owner, self.ctx.get(SolomonToken.name))

        yt.config['token'] = yt_token
        yt.config['proxy']['url'] = 'hahn.yt.yandex.net'

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        env_ = self.ctx.get(Environment_.name)

        solomon_common_labels = {
            'project': 'geosearch_social_links',
            'cluster': 'org_social_links',
            'service': env_,
        }

        if self.ctx['send_stats']:
            sources_stats = dict()
            for source in self.ctx.get(Sources.name):
                source_name = source.split('/')[-1]
                timestamp_str = yt.get_attribute(source, 'modification_time')
                timestamp = datetime.strptime(timestamp_str, '%Y-%m-%dT%H:%M:%S.%fZ')
                sources_stats[source_name] = {
                    "last_update": (datetime.utcnow() - timestamp).total_seconds() / 60.0 / 60.0,
                    "count": yt.table_commands.row_count(source)
                }
            solomon_common_labels['host'] = 'sources'
            push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors(sources_stats))

        # merge update from aurora
        if self.ctx.get(AuroraMerger.name):
            merger = self.sync_resource(self.ctx.get(AuroraMerger.name))
            merged_path = self.full_profiles_dir + {'stable': 'prod', 'testing': 'testing'}[env_]
            storage_path = self.full_profiles_dir + 'archive_' + {'stable': 'prod', 'testing': 'testing'}[env_]
            merger_command = [
                merger,
                merged_path,
                storage_path,
                '//home/extdata/aurora/social_network/other/vk_com_groups',
                '//home/extdata/aurora/social_network/other/facebook_posts',
                '--out_stats', 'merger_stats.json',
            ]

            logging.info('Run command: {}'.format(' '.join(merger_command)))
            run_process(merger_command, stderr=open(os.path.join(get_logs_folder(), 'aurora_merger.stderr'), 'w'))
            logging.info('Done')

            if self.ctx['send_stats']:
                solomon_common_labels['host'] = 'aurora_merger'
                push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors_from_file('merger_stats.json'))
            self.create_resource(
                description='AuroraMerger stats',
                resource_path='merger_stats.json',
                resource_type=resource_types.TASK_CUSTOM_LOGS,
            )

        # extract facts from merged table
        if self.ctx.get(FactExtractor.name):
            fact_extractor = self.sync_resource(self.ctx.get(FactExtractor.name))
            fact_extractor_command = [
                fact_extractor,
                '--proxy', 'hahn.yt.yandex.net',
                merged_path,
                '--out_stats', 'extractor_stats.json',
            ]

            logging.info('Run command: {}'.format(' '.join(fact_extractor_command)))
            run_process(fact_extractor_command, stderr=open(os.path.join(get_logs_folder(), 'fact_extractor.stderr'), 'w'))
            logging.info('Done')

            if self.ctx['send_stats']:
                solomon_common_labels['host'] = 'fact_extractor'
                push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors_from_file('extractor_stats.json'))
            self.create_resource(
                description='FactExtractor stats',
                resource_path='extractor_stats.json',
                resource_type=resource_types.TASK_CUSTOM_LOGS,
            )

        # upload images to mds
        if self.ctx.get(ImagesMdsUploader.name):
            images_uploader = self.sync_resource(self.ctx.get(ImagesMdsUploader.name))
            images_uploader_command = [
                images_uploader,
                '--proxy', 'hahn.yt.yandex.net',
                merged_path,
                '--out_stats', 'uploader_stats.json',
            ]

            logging.info('Run command: {}'.format(' '.join(images_uploader_command)))
            run_process(
                images_uploader_command,
                stderr=open(os.path.join(get_logs_folder(), 'images_uploader.stderr'), 'w')
            )
            logging.info('Done')

            if self.ctx['send_stats']:
                solomon_common_labels['host'] = 'mds_uploader'
                push_to_solomon_v2(solomon_token, solomon_common_labels, self.create_sensors_from_file('uploader_stats.json'))
            self.create_resource(
                description='ImagesMdsUploader stats',
                resource_path='uploader_stats.json',
                resource_type=resource_types.TASK_CUSTOM_LOGS,
            )

        # backup merged table
        if self.ctx.get(YtBackuper.name):
            yt_backuper = self.sync_resource(self.ctx.get(YtBackuper.name))
            yt_backuper_command = [yt_backuper,
                                   '--backup_dir', self.full_profiles_dir + 'backups',
                                   '--proxy', 'hahn.yt.yandex.net',
                                   '--displacement_period', '8',
                                   merged_path]
            logging.info('Run command: {}'.format(' '.join(yt_backuper_command)))
            run_process(yt_backuper_command, stderr=open(os.path.join(get_logs_folder(), 'yt_backuper.stderr'), 'w'))
            logging.info('Done')

        # build table with reduced profiles that contain only actual posts
        if self.ctx.get(ActualProfilesBuilder.name):
            actual_profiles_builder = self.sync_resource(self.ctx.get(ActualProfilesBuilder.name))
            actual_profiles_path = self.profiles_dir + {'stable': 'prod', 'testing': 'testing'}[env_]
            actual_profiles_builder_command = [actual_profiles_builder,
                                               merged_path,
                                               actual_profiles_path,
                                               '--export_period', self.ctx.get(ExportPeriod.name),
                                               '--max_post_age', self.ctx.get(MaxPostAge.name)]

            logging.info('Run command: {}'.format(' '.join(actual_profiles_builder_command)))
            run_process(actual_profiles_builder_command, stderr=open(os.path.join(get_logs_folder(), 'actual_profiles_builder.stderr'), 'w'))
            logging.info('Done')

        create_misc_resource_and_dir(self.ctx, 'fake_result_resource', 'Empty result', 'result_dir', ORG_EVENTS_UPDATER_RESULT)


__Task__ = Task
