# -*- coding: utf-8 -*-

import os

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.projects.common.base_search_quality import settings as bss
from sandbox.projects.common import error_handlers as eh
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters as sp


class IdxQueryExecutableParameter(sp.ResourceSelector):
    """
        Бинарь idx_query
    """
    name = 'executable_resource_id'
    description = 'idx_query binary'
    resource_type = resource_types.IDX_QUERY_EXECUTABLE
    required = True


class SearchDatabaseParameter(sp.ResourceSelector):
    """
        Поисковый шард
    """
    name = 'search_database_resource_id'
    description = 'Search database'
    resource_type = resource_types.SEARCH_DATABASE
    required = True


class PureParameter(sp.ResourceSelector):
    """
        Данные Pure
    """
    name = 'pure_resource_id'
    description = 'Pure data'
    resource_type = resource_types.PURE_DATA
    required = True


_testing_modes = 'Testing options'


class RecheckResultsParameter(sp.SandboxBoolParameter):
    """
        Запускать процесс дважды для сверки результатов
        (в целях проверки нестабильности)
    """
    name = 'recheck_results'
    description = 'Recheck results'
    default_value = True
    group = _testing_modes


class PanteraFactors(SandboxTask):
    """
        Запускает idx_query в различных режимах на указанном поисковом шарде и pure.
        Сохраняет выхлоп в виде ресурса.
        Документация, режимы: SEARCH-1176
    """

    type = 'PANTERA_FACTORS'

    execution_space = bss.RESERVED_SPACE
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = (
        IdxQueryExecutableParameter,
        SearchDatabaseParameter,
        PureParameter,
        RecheckResultsParameter,
    )

    def on_enqueue(self):
        out_resource = self.create_resource(
            self.descr,
            "idx_query_responses.txt",
            resource_types.IDX_QUERY_OUTPUT,
        )
        self.ctx['out_resource_id'] = out_resource.id

    def on_execute(self):
        os.chdir(self.abs_path())

        self.idx_query_name = self.sync_resource(self.ctx[IdxQueryExecutableParameter.name])
        pure_name = self.sync_resource(self.ctx[PureParameter.name])
        db_name = self.sync_resource(self.ctx[SearchDatabaseParameter.name])

        out_file = self.abs_path() + '/idx_query_responses.txt'

        var_args = [
            ['-w', '1 2', '-f1'],
            ['-w', '1 2', '-f1', '-b'],
            ['-w', '1 2', '-f1', '-q'],
            ['-w', '1 2', '-f1', '-q', '-b'],
            ['-d', '10-20', '-f1', '-s'],
            ['-w', '1 2', '-f3', '-p', pure_name, '-c'],
            ['-w', '1 2', '-f3', '-p', pure_name, '-c', '-b'],
            ['-w', '1 2', '-f3', '-p', pure_name, '-c', '-q'],
            ['-w', '1 2', '-f3', '-p', pure_name, '-c', '-q', '-b'],
            ['-d', '10-20', '-f3', '-p', pure_name, '-c', '-s'],
        ]
        var_args = [
            ['-i', db_name] + args
            for args in var_args
        ]
        for args in var_args:
            # quote args, for shell
            args = ['"{}"'.format(arg) for arg in args]
            self._run_idx_query(
                args=args,
                pure_name=pure_name,
                db_name=db_name,
                result_name=out_file,
            )

    def _run_idx_query(
        self, args, pure_name, db_name, result_name
    ):
        """
            Executes idx_query
        """
        full_args = [
            self.idx_query_name,
        ]
        full_args += args
        full_args += [
            '>>', result_name,
        ]
        str_args = [str(arg) for arg in full_args]
        str_args = ' '.join(str_args)
        result_args = (
            str_args
            .replace(self.idx_query_name, '{binary}')
            .replace(db_name, '{db}')
            .replace(pure_name, '{pure}')
            .replace(result_name, '{output}')
        )

        with open(result_name, 'a') as out:
            out.write('=' * 80 + '\n')
            out.write(result_args + '\n')
            out.write('-' * 80 + '\n')

        proc = process.run_process(
            full_args,
            log_prefix='_run_idx_query',
            check=False,
            shell=True,
            outputs_to_one_file=False,
        )
        if proc.returncode != 0:
            self.set_info('idx_query failed run args:\n' + str_args)
            eh.check_failed('idx_query run failed')


__Task__ = PanteraFactors
