# -*- coding: utf-8 -*-

import os
import time
import shutil
import logging
from threading import Thread
from sandbox.common.types.client import Tag

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths
from sandbox.common.types import task as ctt
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import dolbilka
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import config_processor
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search import requester
from sandbox.projects.common.noapacheupper import search_component as nsc
from sandbox.projects.common.noapacheupper import request as nr
from sandbox.projects.release_machine.components.configs import middle
from sandbox.projects.release_machine.components.configs import upper
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.common.noapacheupper.standalone import GrpcClientExecutable
from sandbox.projects.websearch.models_proxy import resources as mprsrc
from sandbox.projects.websearch.models_proxy import search_component as mpsc


class ParallelComponent(object):
    MIDDLE = middle.MiddleCfg.name
    UPPER = upper.UpperCfg.name
    MODELS_PROXY = "models_proxy"

    params = {
        MIDDLE: [
            cmu.create_resource_params(
                n=1,
                group_name='Baseline (first) middlesearch',
                use_int=True,
                with_evlogdump=False,
            ),
            cmu.create_resource_params(
                n=2,
                group_name='Experimental (second) middlesearch',
                use_int=True,
                with_evlogdump=False,
            ),
        ],

        UPPER: [
            nsc.create_noapacheupper_params(
                n=1,
                neh_cache_mode='read',
                settings=False,
                add_requests=True,
                config_params=True,
            ),
            nsc.create_noapacheupper_params(
                n=2,
                neh_cache_mode='read',
                settings=False,
                add_requests=True,
                config_params=True,
            ),
        ],

        MODELS_PROXY: [
            mpsc.create_models_proxy_params(n=1),
            mpsc.create_models_proxy_params(n=2),
        ],

        "YDO": []
    }


shooting_params = cmu.create_shooting_params()


class RunProfilerParameter(sp.SandboxBoolParameter):
    name = "run_profiler"
    description = "Run profiler (gperftools)"
    default_value = False


class ClearEventLogs(sp.SandboxBoolParameter):  # todo: remove it after SEARCH-6291 resolving
    name = "clear_event_logs"
    description = "Clear event logs (debug SEARCH-6291)"
    default_value = True


class ComponentName(sp.SandboxStringParameter):
    name = "component_name"
    description = "Search component name"
    choices = [(k.capitalize(), k) for k in ParallelComponent.params.keys()]
    default_value = ParallelComponent.MIDDLE
    sub_fields = {
        ParallelComponent.MIDDLE: [
            p.name for p_group in ParallelComponent.params[ParallelComponent.MIDDLE] for p in p_group.params
        ],
        ParallelComponent.UPPER: [
            p.name for p_group in ParallelComponent.params[ParallelComponent.UPPER] for p in p_group.params
        ] + [GrpcClientExecutable.name],
        ParallelComponent.MODELS_PROXY: [
            p.name for p_group in ParallelComponent.params[ParallelComponent.MODELS_PROXY] for p in p_group.params
        ],
    }


class RequestsTimeoutMilliseconds(sp.SandboxIntegerParameter):
    """
        Timeout for requests from middle to base. Will be set for all stages ("allfactors", "info", "reask", "fetch" and "aux") except "nexttier"
    """
    name = "requests_timeout"
    description = "Requests timeout (in milliseconds)"
    group = "Other"
    default_value = None


class TimeoutConfigPatcher(config_processor.ParamPatcherByGroup):
    def __init__(self, timeout):
        self.timeout = timeout

    def patch_param(self, param_name, old_value, group_values):
        # nexttier не включён, так как у него стандартный timeout и так маленький (60ms)
        names_to_patch = ["allfactors", "info", "reask", "fetch", "aux"]
        if (
                param_name == "TimeoutTable" and
                "Name" in group_values and
                group_values["Name"] in names_to_patch
        ):
            return self.timeout
        else:
            return old_value


class StressLimiterSemaphore(sp.SandboxStringParameter):
    name = 'stress_limiter_semaphore'
    description = 'name of semaphore to limit concurrent running tasks'


class StressLimiterCapacity(sp.SandboxIntegerParameter):
    name = 'stress_limiter_capacity'
    description = 'maximum number of concurrent running tasks with same stress_limiter_semaphore'


class ParallelDumpEventlog(bugbanner.BugBannerTask):

    type = 'PARALLEL_DUMP_EVENTLOG'

    input_parameters = (
        shooting_params.params + (
            RunProfilerParameter,
            ClearEventLogs,
            ComponentName,
        ) +
        ParallelComponent.params[ParallelComponent.MIDDLE][0].params +
        ParallelComponent.params[ParallelComponent.MIDDLE][1].params +
        ParallelComponent.params[ParallelComponent.UPPER][0].params +
        ParallelComponent.params[ParallelComponent.UPPER][1].params +
        ParallelComponent.params[ParallelComponent.MODELS_PROXY][0].params +
        ParallelComponent.params[ParallelComponent.MODELS_PROXY][1].params +
        (
            GrpcClientExecutable,
            RequestsTimeoutMilliseconds,
            StressLimiterSemaphore,
            StressLimiterCapacity,
        )
    )
    client_tags = Tag.LINUX_PRECISE & Tag.INTEL_E5_2650
    execution_space = 100 * 1024  # 100 Gb
    required_ram = 130 << 10

    _component_name = None
    _port = None
    _is_int = [None, None]
    _binary = [None, None]
    _config = [None, None]
    _middle_index = [None, None]
    _data = [None, None]
    _pure_dir = [None, None]
    _rearrange_dir = [None, None]
    _rearrange_dynamic_dir = [None, None]
    _rearrange_data_fast_dir = [None, None]
    _reqs_file = ["", ""]
    _archive_model = [None, None]
    _mp_mocker_binary = [None, None]
    _mp_fetchdocdata = [None, None]
    _mp_dyntabledata = [None, None]

    _work_dir = None
    _filled_cache = None
    _search_component_process = None
    _reqs_count = None
    _rps = None
    _n_runs = None
    _n_repeats = None
    _plan_file = None
    _nocache_plan_file = None
    _eventlog_file = None
    _loadlog_file = None
    _cache_dir = None

    def initCtx(self):
        self.ctx['kill_timeout'] = 8 * 60 * 60  # 8 hours

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        n_runs = self.ctx[shooting_params.Nruns.name]
        n_repeats = self.ctx[shooting_params.Nrepeats.name]
        self._component_name = utils.get_or_default(self.ctx, ComponentName)
        if self._component_name == ParallelComponent.UPPER:
            self.required_ram = 250 << 10  # 250Gb ram for noapache
        elif self._component_name == ParallelComponent.MODELS_PROXY:
            self.required_ram = 10 << 10

        self.ctx[cmu.KEY_OUT_EVENTLOGS] = [{}, {}]
        for i in xrange(2):
            d = self.ctx[cmu.KEY_OUT_EVENTLOGS][i]
            for i_run in xrange(n_runs):
                for i_repeat in xrange(n_repeats if n_runs == 1 + i_run else 1):
                    suffix = ['middle1st', 'middle2nd'][i]
                    res_descr = '{0}, session {1}, attempt {2}, {3}'.format(self.descr, i_run, i_repeat, suffix)
                    res_file = 'eventlog.{0}.{1}.{2}'.format(i_run, i_repeat, suffix)
                    resource = self._create_resource(res_descr, res_file, resource_types.EVENTLOG_DUMP)
                    d["{},{}".format(i_run, i_repeat)] = resource.id

        if self.ctx.get(StressLimiterSemaphore.name) and self.ctx.get(StressLimiterCapacity.name):
            self.semaphores(ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(
                    name=self.ctx[StressLimiterSemaphore.name],
                    capacity=self.ctx[StressLimiterCapacity.name])]))

    def on_execute(self):
        self._component_name = utils.get_or_default(self.ctx, ComponentName)
        if self._component_name == ParallelComponent.MIDDLE:
            self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)
        elif self._component_name == ParallelComponent.UPPER:
            self.add_bugbanner(bugbanner.Banners.NoapacheUpper)

        self.do_prepare()
        n_runs = self.ctx[shooting_params.Nruns.name]
        do_restart = self.ctx[shooting_params.RestartMode.name]

        if self.ctx[shooting_params.CacheCompatibilityMode.name]:
            use_filled_cache = True
            self.fill_cache()
        else:
            use_filled_cache = False

        if not do_restart:
            self.start_search_component(use_filled_cache)

        for i_run in xrange(n_runs):
            if do_restart:
                self.start_search_component(use_filled_cache)

            self.do_run(i_run)

            if do_restart:
                self.stop_search_component()
                self.erase_cache()

        if not do_restart:
            self.stop_search_component()

    def do_prepare(self):
        self._port = [
            sc.DEFAULT_MIDDLESEARCH_PORT,
            sc.DEFAULT_MIDDLESEARCH_PORT + 1024,
        ]
        self.create_dirs()
        self.prepare_common_paths()
        if self._component_name == ParallelComponent.MIDDLE:
            self.prepare_middle_paths()
            self.make_plans()
        elif self._component_name == ParallelComponent.UPPER:
            self.prepare_upper_paths()
            self.prepare_config_params()
        elif self._component_name == ParallelComponent.MODELS_PROXY:
            self.prepare_models_proxy_paths()
            self.make_models_proxy_plans()
        self.create_search()

    def do_warmup(self):
        # FIXME(mvel): temp hack (disable warmup)
        if self._component_name != ParallelComponent.MIDDLE:
            return

        warmup_request_count = self.ctx[shooting_params.WarmupRequestCount.name]

        if self.ctx[shooting_params.SequentialShooting.name]:
            for i in xrange(2):
                warmup_thread = _run_dolbilka_thread(
                    self._nocache_plan_file[i],
                    self._search_component_process[i],
                    'warmup_{}'.format(i),
                    warmup_request_count,
                    '&pron=break_cache_for_warmup',
                )
                warmup_thread.join()
        else:
            dolbilka_threads = []
            for i in xrange(2):
                dolbilka_threads.append(_run_dolbilka_thread(
                    self._nocache_plan_file[i],
                    self._search_component_process[i],
                    'warmup_{}'.format(i),
                    warmup_request_count,
                    '&pron=break_cache_for_warmup',
                ))
            for t in dolbilka_threads:
                t.join()

        self.clear_eventlog()

    def fill_cache(self):
        if os.path.exists(self._filled_cache):
            return

        self.start_search_component(use_filled_cache=False)

        plan_file = self._plan_file[0]
        dolbilka_thread = _run_dolbilka_thread(
            plan_file,
            self._search_component_process[0],
            'fill_cache',
            self._reqs_count
        )
        dolbilka_thread.join()

        self.stop_search_component()
        self.clear_eventlog()

        shutil.copytree(self._cache_dir[0], self._filled_cache)
        self.erase_cache()

    def do_shoot(self, i_run, i_repeat, use_nocache_plan, dry_run=False):
        logging.info('repeat # %s.%s', i_run, i_repeat)

        order = range(2)
        if self.ctx[shooting_params.ShuffleShootingOrder.name] and (i_run + i_repeat) % 2 == 1:
            order.reverse()

        if utils.get_or_default(self.ctx, shooting_params.SequentialShooting):
            for i in order:
                plan_file = self._nocache_plan_file[i] if use_nocache_plan else self._plan_file[i]
                dolbilka_thread = _run_dolbilka_thread(
                    plan_file,
                    self._search_component_process[i],
                    'shot_{}_{}_{}'.format(i_run, i_repeat, i),
                    self._reqs_count
                )
                dolbilka_thread.join()
        else:
            dolbilka_threads = []
            for i in order:
                plan_file = self._nocache_plan_file[i] if use_nocache_plan else self._plan_file[i]
                if self._component_name in (ParallelComponent.MIDDLE, ParallelComponent.MODELS_PROXY):
                    logging.info("Using dolbilka")
                    dolbilka_threads.append(_run_dolbilka_thread(
                        plan_file,
                        self._search_component_process[i],
                        'shot_{}_{}_{}'.format(i_run, i_repeat, i),
                        self._reqs_count
                    ))
                else:
                    logging.info("Using grpc client")
                    dolbilka_threads.append(_run_grpc_thread(
                        requests=self._reqs_file[i],
                        component=self._search_component_process[i],
                        session_name='shot_{}_{}_{}'.format(i_run, i_repeat, i),
                        grpc_client_path=self._grpc_client_path,
                        rps=self._rps,
                        requests_limit=self._reqs_count,
                        task_context=self.ctx,
                    ))
                time.sleep(3)

            for t in dolbilka_threads:
                t.join()

        # SyncPageCacheBackend for eventlog keeps a tail of data in memory
        # and flushes everything only on shutdown and reopenlog.
        # Issue reopenlog request to flush data.
        if self._component_name == ParallelComponent.MODELS_PROXY:
            for sc_process in self._search_component_process:
                sc_process.fetch('/admin?action=reopenlog')

        if not dry_run:
            self.store_resources(i_run, i_repeat)
        self.clear_eventlog()

    def do_run(self, i_run):
        logging.info('run # %s', i_run)

        n_runs = self.ctx[shooting_params.Nruns.name]
        n_repeats = self.ctx[shooting_params.Nrepeats.name] if n_runs == 1 + i_run else 1
        nocache = self.ctx[shooting_params.NoCacheMode.name]
        cache = self.ctx[shooting_params.CacheMode.name]

        if cache:
            self.do_shoot(i_run, 0, False, dry_run=True)  # fill cache
            self.do_shoot(i_run, 0, False, dry_run=False)  # real run
        else:
            for i_repeat in xrange(n_repeats):
                use_nocache_plan = True if (nocache and i_repeat == 0) else False
                self.do_shoot(i_run, i_repeat, use_nocache_plan)

    def create_dirs(self):
        base_workdir = self.abs_path('workdir')
        paths.make_folder(base_workdir, delete_content=True)
        self._work_dir = [
            os.path.join(base_workdir, 'middle1st'),
            os.path.join(base_workdir, 'middle2nd')
        ]
        self._filled_cache = os.path.join(base_workdir, 'filled_cache')

        for d in self._work_dir:
            os.mkdir(d)

    def _get_resource_path(self, param_class):
        resource_id = utils.get_or_default(self.ctx, param_class)
        if not resource_id:
            return None
        return self._read_resource(resource_id).abs_path()

    def _get_resource_type(self, param_class):
        resource_id = utils.get_or_default(self.ctx, param_class)
        if not resource_id:
            return None
        return sdk2.Resource[resource_id].type

    def prepare_upper_paths(self):
        grpc_client_res_id = utils.get_or_default(self.ctx, GrpcClientExecutable)
        logging.debug("Got grpc client: %s", grpc_client_res_id)
        self._grpc_client_path = self.sync_resource(grpc_client_res_id)
        os.chmod(self._grpc_client_path, 0o755)
        for i in xrange(2):
            resource_params = ParallelComponent.params[ParallelComponent.UPPER][i]

            if not self._binary[i]:
                self.ctx[resource_params.Binary.name] = apihelpers.get_last_released_resource(
                    upper_resources.NoapacheUpper
                ).id
                self._binary[i] = self.sync_resource(self.ctx[resource_params.Binary.name])

            if not self._config[i]:
                self.ctx[resource_params.Config.name] = utils.get_and_check_last_resource_with_attribute(
                    resource_types.NOAPACHEUPPER_CONFIG,
                    attr_name="autoupdate-resources-noapacheupper-web_kubr-trunk-task-id"
                ).id
                self._config[i] = self.sync_resource(self.ctx[resource_params.Config.name])

            self._rearrange_dir[i] = self._get_resource_path(resource_params.RearrangeData)
            if not self._rearrange_dir[i]:
                if not self._data[i]:
                    self.ctx[resource_params.RearrangeData.name] = apihelpers.get_last_released_resource(
                        resource_types.REARRANGE_DATA
                    ).id
                    self._rearrange_dir[i] = self.sync_resource(self.ctx[resource_params.RearrangeData.name])
                else:
                    self._rearrange_dir[i] = os.path.join(self._data[i], 'rearrange')

            self._rearrange_dynamic_dir[i] = self._get_resource_path(resource_params.RearrangeDynamicData)
            if not self._rearrange_dynamic_dir[i]:
                if not self._data[i]:
                    self.ctx[resource_params.RearrangeDynamicData.name] = apihelpers.get_last_released_resource(
                        resource_types.REARRANGE_DYNAMIC_DATA
                    ).id
                    self._rearrange_dynamic_dir[i] = self.sync_resource(self.ctx[resource_params.RearrangeDynamicData.name])
                else:
                    self._rearrange_dynamic_dir[i] = os.path.join(self._data[i], 'rearrange.dynamic')

            self._rearrange_data_fast_dir[i] = self._get_resource_path(resource_params.RearrangeDataFast)
            if not self._rearrange_data_fast_dir[i]:
                if not self._data[i]:
                    self.ctx[resource_params.RearrangeDataFast.name] = apihelpers.get_last_released_resource_with_attr(
                        upper_resources.RearrangeDataFast, attrs={'vertical': 'web'},
                    ).id
                    self._rearrange_data_fast_dir[i] = self.sync_resource(self.ctx[resource_params.RearrangeDataFast.name])
                else:
                    self._rearrange_data_fast_dir[i] = os.path.join(self._data[i], 'rearrange.fast')

    def prepare_middle_paths(self):
        for i in xrange(2):
            resource_params = ParallelComponent.params[ParallelComponent.MIDDLE][i]
            self._archive_model[i] = self._get_resource_path(resource_params.ArchiveModel)
            self._middle_index[i] = self._get_resource_path(resource_params.Index)
            if self._data[i]:
                if self._get_resource_type(resource_params.Data) == 'QUICK_REARRANGE_RULES_DATA' or self._get_resource_type(resource_params.Data) == 'MIDDLESEARCH_GEO_DATA':
                    self._rearrange_dir[i] = self._data[i]
                else:
                    self._rearrange_dir[i] = os.path.join(self._data[i], 'rearrange')
                self._pure_dir[i] = os.path.join(self._data[i], 'pure')
            self._is_int[i] = utils.get_or_default(self.ctx, resource_params.UseInt)

    def prepare_models_proxy_paths(self):
        for i in xrange(2):
            resource_params = ParallelComponent.params[ParallelComponent.MODELS_PROXY][i]
            self._mp_mocker_binary[i] = self._get_resource_path(resource_params.SubSourcesMockerBinary)
            self._mp_fetchdocdata[i] = self._get_resource_path(resource_params.MockedFetchDocData)
            self._mp_dyntabledata[i] = self._get_resource_path(resource_params.MockedDynTableResults)

    def prepare_common_paths(self):
        for i in xrange(2):
            resource_params = ParallelComponent.params[self._component_name][i]

            self._binary[i] = self._get_resource_path(resource_params.Binary)
            self._config[i] = self._get_resource_path(resource_params.Config)
            self._data[i] = self._get_resource_path(resource_params.Data)
            self._reqs_file[i] = self._get_resource_path(resource_params.Requests)

        self._reqs_count, self._rps, self._n_runs, self._n_repeats = [
            self.ctx[name] for name in [
                shooting_params.ReqCount.name, shooting_params.Rps.name,
                shooting_params.Nruns.name, shooting_params.Nrepeats.name
            ]
        ]

        self._plan_file = [
            os.path.join(self._work_dir[0], 'plan.bin.1st'),
            os.path.join(self._work_dir[1], 'plan.bin.2nd')
        ]
        self._nocache_plan_file = [
            os.path.join(self._work_dir[0], 'nocache_plan.bin.1st'),
            os.path.join(self._work_dir[1], 'nocache_plan.bin.2nd')
        ]
        self._eventlog_file = [
            os.path.join(self._work_dir[0], 'eventlog.bin.1st'),
            os.path.join(self._work_dir[1], 'eventlog.bin.2nd')
        ]
        self._loadlog_file = [
            os.path.join(self._work_dir[0], 'load.log.1st'),
            os.path.join(self._work_dir[1], 'load.log.2nd')
        ]
        self._cache_dir = [
            os.path.join(self._work_dir[0], 'query_cache'),
            os.path.join(self._work_dir[1], 'query_cache')
        ]

    def make_plans(self):
        if self.ctx[shooting_params.ReverseRequests.name]:
            self.do_reverse_requests()

        for i in xrange(2):
            dolbilka.convert_queries_to_plan(
                self._reqs_file[i],
                self._plan_file[i],
                alter_query=lambda q: q + '&nofastcache=1',
                rps=self._rps,
                max_queries=self._reqs_count
            )
            dolbilka.convert_queries_to_plan(
                self._reqs_file[i],
                self._nocache_plan_file[i],
                alter_query=lambda q: q + '&nofastcache=1&nocache=da',
                rps=self._rps,
                max_queries=self._reqs_count,
            )

    def make_models_proxy_plans(self):
        if self.ctx[shooting_params.ReverseRequests.name]:
            self.do_reverse_requests()

        patcher_res_id = apihelpers.get_last_resource(mprsrc.ModelsProxyRequestsPatcher)
        patcher = self.sync_resource(patcher_res_id)

        for i in xrange(2):
            request_patches_param = ParallelComponent.params[ParallelComponent.MODELS_PROXY][i].RequestPatches
            tmp_file_name = 'patched-queries.tsv'
            patcher_args = [
                patcher,
                '--remove-log-params',
                '--input', self._reqs_file[i],
                '--output', tmp_file_name,
                '--generate-timestamps',
                '--rps', str(self._rps),
            ]
            for p in utils.get_or_default(self.ctx, request_patches_param) or []:
                patcher_args.append('--set-param')
                patcher_args.append(p)

            process.run_process(patcher_args)
            dolbilka.convert_queries_to_plan(
                tmp_file_name,
                self._plan_file[i],
                alter_query=lambda q: '/models_proxy\t\tPOST\t' + q,
                rps=self._rps,
                max_queries=self._reqs_count
            )

            process.run_process(patcher_args + ['--set-param', 'nocache=1'])
            dolbilka.convert_queries_to_plan(
                tmp_file_name,
                self._nocache_plan_file[i],
                alter_query=lambda q: '/models_proxy\t\tPOST\t' + q,
                rps=self._rps,
                max_queries=self._reqs_count,
            )

    def prepare_config_params(self):
        self._config_params = []
        for i in xrange(2):
            params = ParallelComponent.params[ParallelComponent.UPPER][i]
            self._config_params.append({
                'Collection/MaxCalcRelevQueueSize': utils.get_or_default(self.ctx, params.MaxCalcRelevQueueSize),
                'Collection/DebugOptions' : utils.get_or_default(self.ctx, params.ConfigDebugOptions),
            })

    def do_reverse_requests(self):
        logging.info('reverse first requests set')

        if self._reqs_count > 0:
            requests_to_reverse = os.path.join(self._work_dir[0], 'cut_requests.txt')
            with open(self._reqs_file[0]) as in_file, open(requests_to_reverse, "w") as out_file:
                process.run_process(['head', '-n', str(self._reqs_count)], stdin=in_file, stdout=out_file)
        else:
            requests_to_reverse = self._reqs_file[0]

        reversed_reqs_file = os.path.join(self._work_dir[0], 'reversed_requests.txt')
        logging.info('reversed set is stored at %s', reversed_reqs_file)
        tail_command = ['tac']
        with open(requests_to_reverse) as in_file, open(reversed_reqs_file, "w") as out_file:
            process.run_process(tail_command, stdin=in_file, stdout=out_file)
        self._reqs_file[0] = reversed_reqs_file

    def create_search(self):
        self._search_component_process = []
        use_profiler = utils.get_or_default(self.ctx, RunProfilerParameter)
        for i in xrange(2):
            if self._component_name == ParallelComponent.MIDDLE:
                config_params = None
                timeout = self.ctx.get(RequestsTimeoutMilliseconds.name)
                if timeout is not None:
                    timeout = "{}ms".format(timeout)
                    config_params = {
                        "Collection/ScatterOptions": TimeoutConfigPatcher(timeout),
                        "Collection/TimeoutTable": timeout,
                    }
                component = sc.Middlesearch(
                    is_int=self._is_int[i],
                    work_dir=self._work_dir[i],
                    binary=self._binary[i],
                    config_file=self._config[i],
                    pure_dir=self._pure_dir[i],
                    rearrange_dir=self._rearrange_dir[i],
                    rearrange_index_dir=self._middle_index[i],
                    archive_model_path=self._archive_model[i],
                    event_log=self._eventlog_file[i],
                    load_log=self._loadlog_file[i],
                    port=self._port[i],
                    patch_queues_size=False,
                    query_cache=self._cache_dir[i],
                    use_profiler=use_profiler,
                    use_gperftools=use_profiler,
                    config_params=config_params,
                )
            elif self._component_name == ParallelComponent.UPPER:
                component = nsc.Noapacheupper(
                    is_int=False,
                    work_dir=self._work_dir[i],
                    binary=self._binary[i],
                    config_file=self._config[i],
                    rearrange_dir=self._rearrange_dir[i],
                    rearrange_dynamic_dir=self._rearrange_dynamic_dir[i],
                    rearrange_data_fast_dir=self._rearrange_data_fast_dir[i],
                    event_log=self._eventlog_file[i],
                    load_log=self._loadlog_file[i],
                    port=self._port[i],
                    patch_queues_size=False,
                    query_cache=None,
                    use_profiler=use_profiler,
                    use_gperftools=use_profiler,
                    # we don't need other modes
                    apphost_mode=True,
                    use_verify_stderr=False,
                    config_params=self._config_params[i],
                )
            elif self._component_name == ParallelComponent.MODELS_PROXY:
                mocker = mpsc.ModelsProxyMocker(
                    binary=self._mp_mocker_binary[i],
                    port=self._port[i]+1,
                    fetchdocdata=self._mp_fetchdocdata[i],
                    dyntabledata=self._mp_dyntabledata[i],
                )
                component = mpsc.ModelsProxy(
                    work_dir=self._work_dir[i],
                    binary=self._binary[i],
                    port=self._port[i],
                    config_file=self._config[i],
                    data_dir=self._data[i],
                    event_log=self._eventlog_file[i],
                    mocker=mocker,
                )

            self._search_component_process.append(component)

    def start_search_component(self, use_filled_cache=True):
        if use_filled_cache:
            logging.info("Fill cache when starting search component")
            for cache_dir in self._cache_dir:
                for f in os.listdir(self._filled_cache):
                    src = os.path.join(self._filled_cache, f)
                    dst = os.path.join(cache_dir, f)
                    shutil.copytree(src, dst)

        for sc_process in self._search_component_process:
            sc_process.start()
            sc_process.wait()

        self.do_warmup()

    def stop_search_component(self):
        for sc_process in self._search_component_process:
            sc_process.stop()

    def clear_eventlog(self):
        if not utils.get_or_default(self.ctx, ClearEventLogs):
            logging.info("EventLog truncation disabled")
            return

        for ef in self._eventlog_file:
            logging.info("Truncating eventlog file '%s'", ef)
            with open(ef, 'w') as f:
                f.truncate(0)

    def erase_cache(self):
        for cache_dir in self._cache_dir:
            paths.make_folder(cache_dir, delete_content=True)

    def store_resources(self, i_run, i_repeat):
        for i in xrange(2):
            log_addition = ['1st', '2nd'][i] + ' middle'

            logging.info('store eventlog {0}_{1}, {2}'.format(i_run, i_repeat, log_addition))
            eventlog_resource = channel.sandbox.get_resource(
                self.ctx[cmu.KEY_OUT_EVENTLOGS][i]["{},{}".format(i_run, i_repeat)]
            )
            logging.info('copying from {0} to {1}'.format(self._eventlog_file[i], eventlog_resource.path))
            shutil.copyfile(self._eventlog_file[i], eventlog_resource.path)
            logging.info('mark resource {0} ready'.format(eventlog_resource.id))
            self.mark_resource_ready(eventlog_resource)


def _get_time():
    return time.strftime("%d %b %Y %H:%M:%S", time.localtime())


class _DolbilkaThread(Thread):
    def __init__(self, d_executor, plan, target, session_name):
        Thread.__init__(self)
        self.d_executor, self.plan, self.target, self.session_name = d_executor, plan, target, session_name

    def run(self):
        self.d_executor.run_session_and_dumper(self.plan, self.target, self.session_name, run_once=True)


def _run_dolbilka_thread(plan, target, session_name, requests_limit, augment_url=''):
    d_executor = dolbilka.DolbilkaExecutor()
    d_executor.mode = dolbilka.DolbilkaExecutorMode.PLAN_MODE
    d_executor.requests = requests_limit
    d_executor.augment_url = augment_url
    dolbilka_thread = _DolbilkaThread(d_executor, plan, target, session_name)
    dolbilka_thread.start()
    return dolbilka_thread


class _RequesterThread(Thread):
    def __init__(self, requests, component, session_name, task_context):
        Thread.__init__(self)
        self.requests = requests
        self.component = component
        self.session_name = session_name
        self.task_context = task_context

    def run(self):
        r = requester.Requester()
        logging.info("Iterating using and42@ requester over %s", self.requests)
        req_iter = nr.binary_requests_iterator(
            self.requests,
            use_apphost=True,
        )
        r.use(req_iter, self.component, ctx=self.task_context, start_stop=False)


def _run_requester_thread(requests, component, session_name, requests_limit=0, augment_url='', task_context=None):
    requester_thread = _RequesterThread(requests, component, session_name, task_context)
    requester_thread.start()
    return requester_thread


class _GRPCThread(Thread):
    def __init__(self, requests, component, session_name, task_context, grpc_client_path, requests_limit, rps):
        Thread.__init__(self)
        self.requests = requests
        self.component = component
        self.session_name = session_name
        self.task_context = task_context
        self.grpc_client_path = grpc_client_path
        self.requests_limit = requests_limit
        self.rps = rps

    def run(self):
        try:
            cmd = "{} localhost:{} -P {} -t bin -s {} -T {}".format(
                self.grpc_client_path, int(self.component.port) + 2, self.requests, self.rps, int(self.requests_limit/self.rps),
            )
            logging.debug("Try to run process: %s", cmd)
            process.run_process(cmd, shell=True, log_prefix="grpc_client_shoot")
        except Exception as err:
            eh.log_exception("Fail in grpc mode", err)


def _run_grpc_thread(requests, component, session_name, grpc_client_path, rps, requests_limit=0, augment_url='', task_context=None):
    requester_thread = _GRPCThread(requests, component, session_name, task_context, grpc_client_path, requests_limit, rps)
    requester_thread.start()
    return requester_thread


__Task__ = ParallelDumpEventlog
