# -*- coding: utf-8 -*-

import os
import re
import json
import sys
import logging

from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, TaskSelector
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.BaseReportShardTask import getClassesByResources
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


source_pattern = 'ARCNEWS_SLAVE_NEWSD'
source_pattern_re = re.compile('\\b{}\\b'.format(source_pattern))
shard_nums = {}
shard_serial = 0
arcshards_name = 'archards.json'


class AppHostConfigTask(TaskSelector):
    name = 'BUILD_APP_HOST_CONFIG_BUNDLE'
    task_type = 'BUILD_APP_HOST_CONFIG_BUNDLE'
    description = "Select AppHost config bundle task (optional)"
    required = False
    default_value = None


class PatchNewsAppHostGraphs(nanny.ReleaseToNannyTask, SandboxTask):
    """
    """
    type = 'PATCH_NEWS_APP_HOST_GRAPHS'

    input_parameters = getClassesByResources(['NEWS_ARCHARDS_EXECUTABLE']) + [AppHostConfigTask] + [SandboxArcadiaUrlParameter]

    def on_execute(self):
        global shard_nums
        global shard_serial
        global arcshards_name

        shard_nums = {}
        shard_serial = 0

        workcopy_dir = self.abs_path('workcopy')
        make_folder(workcopy_dir)
        os.chdir(workcopy_dir)

        if not self.ctx.get(AppHostConfigTask.name) and not self.ctx.get("app_host_config_bundle_task_id"):
            context = self.ctx.copy()
            context["choose_vertical"] = "NEWS"
            context["build_service_resources"] = False
            context["no_backends_in_graph"] = True
            context["enrich_graphs_revision"] = True

            subtask = self.create_subtask(
                task_type="BUILD_APP_HOST_CONFIG_BUNDLE",
                description=self.descr,
                input_parameters=context,
                priority=self.priority,
                inherit_notifications=True
            )

            self.ctx["app_host_config_bundle_task_id"] = subtask.id
            logging.info(
                "Starting subtask BUILD_APP_HOST_CONFIG_BUNDLE {}".format(subtask)
            )

            self.wait_tasks(
                [subtask],
                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                 self.Status.EXCEPTION, self.Status.TIMEOUT),
                True
            )

        AppHostConfigBundleTaskId = self.ctx.get("app_host_config_bundle_task_id") or self.ctx.get(AppHostConfigTask.name)
        logging.info(
            "Using AppHostConfigBundle task with id {}".format(AppHostConfigBundleTaskId)
        )
        if AppHostConfigBundleTaskId is None:
            raise SandboxTaskFailureError("No apphost config bundle provided and some error with automatic subtask happend")

        AppHostConfigBundleResource = channel.sandbox.list_resources(resource_type="APP_HOST_CONFIG_BUNDLE_NEWS",
                                                                     task_id=AppHostConfigBundleTaskId,
                                                                     status="READY")[0]

        AppHostBackendsResource = channel.sandbox.list_resources(resource_type="APP_HOST_BACKENDS_CONFIG_NEWS",
                                                                 task_id=AppHostConfigBundleTaskId,
                                                                 status="READY")[0]

        logging.info(
            "Using resources {} and {} for executing".format(AppHostConfigBundleResource, AppHostBackendsResource)
        )

        path = self.sync_resource(AppHostConfigBundleResource)
        graph_tar_name = os.path.basename(path)
        graph_dir_name = re.sub(r'\.tar\.gz$', '', graph_tar_name)

        run_process('tar zxf "{}"'.format(path), shell=True)

        if os.path.exists(graph_tar_name):
            os.remove(graph_tar_name)

        path = self.sync_resource(self.ctx.get('res_news_archards_executable'))

        run_process('{} > "{}"'.format(path, arcshards_name), shell=True)

        archards_file = open(arcshards_name, 'r')
        archards = json.loads(archards_file.read())
        archards_file.close()

        graphs_dir = os.path.join(graph_dir_name, '_ALL', 'graphs')
        if not os.path.exists(graphs_dir):
            raise Exception("Dir " + graphs_dir + " not found. Did you just tried to build backends inside graphs?")

        for graph_name in os.listdir(graphs_dir):
            patch_graph(os.path.join(graphs_dir, graph_name), archards['man'])

        run_process('tar zcf "{}" "{}"'.format(graph_tar_name, graph_dir_name), shell=True)

        resource = self.create_resource(
            description=self.descr,
            resource_path='workcopy/{}'.format(graph_tar_name),
            resource_type=resource_types.APP_HOST_CONFIG_BUNDLE_NEWS,
            arch=self.arch
        )

        backends_path = self.sync_resource(AppHostBackendsResource)
        patch_inverse_backends(backends_path, archards)

        resource_backends = self.create_resource(
            description=self.descr,
            resource_path='workcopy/{}'.format('sharded_backends.json'),
            resource_type=app_host_resources.APP_HOST_BACKENDS_CONFIG_NEWS,
            arch=self.arch
        )

        self.mark_resource_ready(resource.id)
        self.mark_resource_ready(resource_backends.id)

    def on_release(self, rest):
        nanny.ReleaseToNannyTask.on_release(self, rest)
        SandboxTask.on_release(self, rest)


def shard_name(source_name):
    global shard_nums
    global shard_serial
    global source_pattern

    if source_name not in shard_nums:
        shard_serial += 1
        shard_nums[source_name] = shard_serial

    return '{}_{}'.format(source_pattern, str(shard_nums[source_name]))


def patch_inverse_backends(backends_path, arcshards):
    global source_pattern

    backends_file = open(backends_path, 'r')
    backends = json.loads(backends_file.read())
    backends_file.close()

    common_backend_info = {
        'balancing_id': source_pattern,
        'balancing_scheme': 'rrobin',
        'location': 'current'
    }

    for location, shards in arcshards.iteritems():
        for shard, hosts in shards.iteritems():
            new_spec = {
                'backend_descrs': []
            }

            for key, value in common_backend_info.iteritems():
                new_spec[key] = value

            for host in hosts:
                host_spec = {
                    'host': host['hostname'],
                    'min_weight': 0.02,
                    'port': host['port'],
                    'protocol': 'post',
                    'weight': 0.5,
                }
                for key in ['ipv6addr', 'ipv4addr']:
                    if key in host:
                        host_spec['ip'] = host[key]
                        break

                new_spec['backend_descrs'].append(host_spec)

            for raw_location, location_description in backends.iteritems():
                if (raw_location.startswith(location.upper()) or (location == 'man' and raw_location.startswith('ALL'))):
                    backends[raw_location][shard_name(shard)] = new_spec

    backends_file = open('sharded_backends.json', 'w')
    backends_file.write(json.dumps(backends, sort_keys=True, indent=4))
    backends_file.close()


def patch_graph(graph_path, shards):
    global source_pattern
    global source_pattern_re

    if not os.path.isfile(graph_path) or not re.search(r'\.json$', graph_path):
        return

    graph_file = open(graph_path, 'r')
    graph = json.loads(graph_file.read())
    graph_file.close()

    for node, dependencies in graph['graph'].iteritems():
        if node == source_pattern:
            continue

        new_dependencies = []

        for dependency in dependencies:
            if dependency == source_pattern:
                for shard in shards.keys():
                    new_dependencies.append(shard_name(shard))

                continue

            new_dependencies.append(dependency)

        graph['graph'][node] = new_dependencies

    if source_pattern in graph['graph']:
        dependencies = graph['graph'][source_pattern]
        del graph['graph'][source_pattern]

        for shard in shards.keys():
            graph['graph'][shard_name(shard)] = dependencies

    if source_pattern in graph['sources']:
        spec = graph['sources'][source_pattern]
        del graph['sources'][source_pattern]
        for shard, hosts in shards.iteritems():
            graph['sources'][shard_name(shard)] = spec

    if 'edge_expressions' in graph:
        new_edge_expressions = {}
        node_delimiter = '->'

        for edge, expression in graph['edge_expressions'].iteritems():
            pair = map(str.strip, edge.encode('utf-8').split(node_delimiter))

            if (len(pair) != 2) or (pair[0] == pair[1]):
                sys.exit(1)

            i = 0
            expanded = False

            for node in pair:
                if node == source_pattern:
                    other_index = 1 - i

                    for shard in shards.keys():
                        new_pair = [None, None]

                        new_pair[other_index] = pair[other_index]
                        new_pair[i] = shard_name(shard)

                        new_edge_expressions[node_delimiter.join(new_pair).decode('utf-8')] = re.sub(source_pattern_re, new_pair[i], expression)

                    expanded = True
                    break

                i += 1

            if not expanded:
                new_edge_expressions[edge] = expression

        graph['edge_expressions'] = new_edge_expressions

    graph_file = open(graph_path, 'w')
    graph_file.write(json.dumps(graph, sort_keys=True, indent=4))
    graph_file.close()


__Task__ = PatchNewsAppHostGraphs
