# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects.common import dolbilka


class PlanParameter(LastReleasedResource):
    name = 'plan_resource_id'
    description = 'Plan'
    resource_type = ['BASESEARCH_PLAN', 'MIDDLESEARCH_PLAN']


class AddCgiParameter(SandboxStringParameter):
    """
        Какой параметр следует добавить к запросам
    """
    name = 'new_cgi_param'
    description = "New cgi param (must be encoded, may contain multiple params divided by ';')"
    required = False
    default_value = ''


class RemoveCgiParameters(SandboxStringParameter):
    """
        Какой параметр следует удалить из запросов
    """
    name = 'remove_cgi_params'
    description = "Remove cgi params (must be encoded, may contain multiple params divided by ';')"
    required = False
    default_value = ''


class ResourceAttrsParameter(SandboxStringParameter):
    """
        Какие атрибуты следует добавить к создаваемому ресурсу
    """
    name = 'resource_attrs'
    description = 'Set attrs to resources (e.g.: attr1=v1, attr2=v2)'


class PatchPlan(SandboxTask):
    """
        Патчит план:
        -достает из него запросы (GET_QUERIES_FROM_PLAN)
        -добавляет/удаляет параметры (PATCH_QUERIES)
        -создает новый план (GENERATE_PLAN_FROM_QUERIES)
    """

    type = 'PATCH_PLAN'

    cores = 1

    input_parameters = [
        PlanParameter,
        AddCgiParameter,
        RemoveCgiParameters,
        ResourceAttrsParameter,
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        self.ctx['out_resource_id'] = self.create_resource(
            self.descr,
            'plan.bin',
            'BASESEARCH_PLAN',
            arch='any'
        ).id

    def on_execute(self):
        cgi_params = utils.get_or_default(self.ctx, AddCgiParameter)
        remove_cgi_params = utils.get_or_default(self.ctx, RemoveCgiParameters)
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        dolbilka.patch_plan(self.ctx[PlanParameter.name], out_resource, cgi_params, remove_cgi_params)

        attributes = self.ctx.get(ResourceAttrsParameter.name)
        utils.set_resource_attributes(self.ctx['out_resource_id'], string.parse_attrs(attributes))


__Task__ = PatchPlan
