# -*- coding: utf-8 -*-

import logging

import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects.common import dolbilka


class QueriesParameter(ResourceSelector):
    name = 'queries_resource_id'
    description = 'Source queries'
    resource_type = [
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS
    ]


class QueriesLimitParameter(SandboxIntegerParameter):
    name = 'queries_limit'
    description = 'Limit number of queries to patch (0 = all)'
    default_value = 0


class AddCgiParameter(SandboxStringParameter):
    """
        Какой параметр следует добавить к запросам
        Если передавать параметры через незаэкранированную ';' - они будут циклически
        дописываться к запросам по одному параметру в запрос.
    """
    name = 'new_cgi_param'
    description = "New cgi params (must be encoded, see task description for details)"
    required = False
    default_value = ''


class RemoveCgiParameters(SandboxStringParameter):
    """
        Какой параметр следует удалить из запросов
    """
    name = 'remove_cgi_params'
    description = "Remove cgi params (must be encoded, may contain multiple params divided by ';')"
    required = False
    default_value = ''


class ResourceAttrsParameter(SandboxStringParameter):
    """
        Какие атрибуты следует добавить к создаваемому ресурсу
    """
    name = 'resource_attrs'
    description = 'Set attrs to resources (e.g.: attr1=v1, attr2=v2)'


class PatchQueries(SandboxTask):
    """
        Create new resource PLAIN_TEXT_QUERIES based on patched source resource PLAIN_TEXT_QUERIES.

        Parameters:

        **AddCgiParameter** (``new_cgi_param``) - encoded cgi-parameters. Please, encode your ';' as '%3B'.
            Otherwise, params will be divided on parts by unencoded ';' and will
            append to queries one by one in cycle.

        **RemoveCgiParameters** (``remove_cgi_params``) - encoded (e.g. ';' should be encoded as %3B)
            cgi-parameters to remove from queries. Use unescaped '``;``' as multiple parameters separator.
            Use wildcard ``*`` character at the end of param for cut prefixes, please consider proper
            regex-escaping in this case.
            Example: ``&pron=termtpsz*;&pron=notermsearch``.
    """

    type = 'PATCH_QUERIES'

    input_parameters = [
        QueriesParameter,
        QueriesLimitParameter,
        AddCgiParameter,
        RemoveCgiParameters,
        ResourceAttrsParameter,
    ]
    execution_space = 15 * 1024  # 15 Gb
    cores = 1

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        source_resource = channel.sandbox.get_resource(self.ctx[QueriesParameter.name])
        self.ctx['out_resource_id'] = self.create_resource(
            self.descr,
            'queries.txt',
            source_resource.type,
            arch='any'
        ).id

    def on_execute(self):
        source_queries_res_id = self.ctx[QueriesParameter.name]
        self.sync_resource(source_queries_res_id)
        source_resource = channel.sandbox.get_resource(source_queries_res_id)
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        number_of_queries = dolbilka.patch_queries(
            source_resource.path,
            out_resource.path,
            utils.get_or_default(self.ctx, AddCgiParameter),
            utils.get_or_default(self.ctx, RemoveCgiParameters),
            utils.get_or_default(self.ctx, QueriesLimitParameter) or None,
        )

        self.set_info("{} queries patched successfully".format(number_of_queries))

        resource_attributes = self.ctx.get(ResourceAttrsParameter.name)
        if resource_attributes:
            logging.info('Set resource attributes %s', resource_attributes)
            for k, v in string.parse_attrs(resource_attributes).iteritems():
                channel.sandbox.set_resource_attribute(self.ctx['out_resource_id'], k, v)


__Task__ = PatchQueries
