import logging
from sandbox import sdk2

import time
import datetime
import os

import sandbox.common.types.task as ctt
from sandbox.projects.PersonalPoiGenerator.Generator import PersonalPoiGeneratePairs
from sandbox.projects.PersonalPoiGenerator.Features import PersonalPoiFeatures
from sandbox.projects.UploadToSaasKv import Uploader
from sandbox.projects.resource_types import DUMP_TO_INDEXERPROXY_EXECUTABLE


class PersonalPoiConveyor(sdk2.Task):
    """Conveyor for cid-oid generation"""

    class Parameters(sdk2.Task.Parameters):
        date = sdk2.parameters.String(
            'Date for calculate',
        )
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String('Your token name in vault', default='yt-token', required=True)
            yt_common_folder = sdk2.parameters.String(
                'Your folder',
                required=True
            )

        with sdk2.parameters.Group('Generator parameters') as g_parameters:
            probability = sdk2.parameters.Float(
                'Org1 probability',
                required=False,
            )

        with sdk2.parameters.Group('SaaS load parameters') as saas_parameters:
            with sdk2.parameters.RadioGroup('Saas indexer') as indexer:
                indexer.values['saas-indexerproxy-maps-prestable.yandex.net'] = indexer.Value(value='Testing',
                                                                                              default=True)
                indexer.values['saas-indexerproxy-maps-kv.yandex.net'] = indexer.Value(value='Stable')
            saas_key = sdk2.parameters.String(
                'Your service key for SaaS',
                required=True,
            )
            saas_dumper = sdk2.parameters.Resource(
                'Dumper to SaaS',
                resource_type=DUMP_TO_INDEXERPROXY_EXECUTABLE,
                required=True,
            )
            saas_tables = sdk2.parameters.String(
                'Your tables for SaaS',
                required=False,
            )

    class Requirements(sdk2.Task.Requirements):
        pass

    def _create_subtask(self, task_type, params, description):
        task = self.server.task({
            'type': task_type,
            'context': params,
            'children': True,
        })
        self.server.task[task['id']].update({
            'description': description,
            'owner': self.owner,
            'notifications': self.server.task[self.id].read()["notifications"],
            'priority': {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
        })
        self.server.batch.tasks.start.update([task['id']])
        return task['id']

    def on_execute(self):
        if self.Parameters.date is None or len(str(self.Parameters.date)) == 0:
            self.Context.day = (datetime.date.today() - datetime.timedelta(days=1)).strftime('%Y-%m-%d')
        else:
            self.Context.day = str(self.Parameters.date)

        with self.memoize_stage.squeeze_phase:
            day = datetime.datetime.strptime(self.Context.day, '%Y-%m-%d').date()
            logging.info('Start squeeze feature')
            squeeze_task = PersonalPoiFeatures(
                self,
                description='Feature phase of regular process {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_vault_token=self.Parameters.yt_vault_token,
                first_day=day.strftime('%Y-%m-%d'),
                last_day=(day + datetime.timedelta(days=1)).strftime('%Y-%m-%d'),
            )
            squeeze_task.enqueue()

            raise sdk2.WaitTask(
                [squeeze_task.id],
                ctt.Status.Group.FINISH,
                wait_all=True
            )

        with self.memoize_stage.generate_day_phase:
            day = datetime.datetime.strptime(self.Context.day, '%Y-%m-%d').date()
            logging.info('Start generating one day')
            generate_task = PersonalPoiGeneratePairs(
                self,
                description='First generation phase of regular process {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                **{
                    'date': (day - datetime.timedelta(days=1)).strftime('%Y-%m-%d'),
                    'yt_vault_token': self.Parameters.yt_vault_token,
                    'common_folder': self.Parameters.yt_common_folder,
                    'is_saas_reduce': False,
                    'do_map': True,
                    'do_fast_pairs': True,
                    'do_fast_bad_pairs': True,
                    'do_fast_zoom_pairs': True,
                    'do_fast_card_pairs': True,
                    'do_fast_mnet_pairs': True,
                    'do_fast_deep_pairs': True,
                    'probability': self.Parameters.probability,
                    'expired_days': 7,

                }
            )
            generate_task.enqueue()

            raise sdk2.WaitTask(
                [generate_task.id],
                ctt.Status.Group.FINISH,
                wait_all=True
            )

        with self.memoize_stage.generate_phase:
            day = datetime.datetime.strptime(self.Context.day, '%Y-%m-%d').date()
            logging.info('Start generating saas tables')
            generate_task = PersonalPoiGeneratePairs(
                self,
                description='Generation phase of regular process {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                **{
                    'date': day.strftime('%Y-%m-%d'),
                    'yt_vault_token': self.Parameters.yt_vault_token,
                    'common_folder': self.Parameters.yt_common_folder,
                    'is_saas_reduce': True,
                    'do_fast_pairs': True,
                    'do_fast_bad_pairs': True,
                    'do_fast_zoom_pairs': True,
                    'do_fast_card_pairs': True,
                    'do_fast_mnet_pairs': True,
                    'do_fast_deep_pairs': True,
                    'expired_days': 7,

                }
            )
            generate_task.enqueue()

            raise sdk2.WaitTask(
                [generate_task.id],
                ctt.Status.Group.FINISH,
                wait_all=True
            )

        with self.memoize_stage.saas_upload_good:
            day = datetime.datetime.strptime(self.Context.day, '%Y-%m-%d').date()
            expired_time_second = time.mktime((day + datetime.timedelta(days=7)).timetuple()) - time.timezone
            expired_time_minutes = int(expired_time_second / 60)
            if self.Parameters.saas_tables is None or len(str(self.Parameters.saas_tables)) == 0:
                return
            result_tables = str(self.Parameters.saas_tables).split(',')
            logging.info('Start uploading')
            uploading_tasks = list()
            table_to_jobs_count = {
                'any-click': '2000',
                'matrixnet': '4000',
            }
            for table in map(lambda name: os.path.join(self.Parameters.yt_common_folder, 'pairs', 'saas', name),
                             result_tables):
                uploading_tasks.append(self._create_subtask(
                    Uploader.type,
                    {
                        'saas_indexer': self.Parameters.indexer,
                        'saas_dumper': self.Parameters.saas_dumper.id,
                        'yt_token_secret': self.Parameters.yt_vault_token,
                        'saas_key': self.Parameters.saas_key,
                        'source_table_path': table,
                        'errors_table': table + '.err',
                        'saas_action': 'atModify',
                        'saas_processor': 'personal_geo',
                        'deadline': str(expired_time_minutes),
                        'mr_jobs_count': table_to_jobs_count.get(os.path.basename(table), '200')

                    },
                    'Upload phase of regular process',
                ))

            raise sdk2.WaitTask(
                uploading_tasks,
                ctt.Status.Group.FINISH,
                wait_all=True
            )
