import datetime
import functools
import json
import logging
import os

from sandbox import sdk2

import sandbox.sandboxsdk.environments as environments
import sandbox.common.types.task as ctt

from sandbox.common.errors import TaskError

from sandbox.projects.PersonalPoiGenerator.UniversalTask import (
    PersonalPoisUniversalTask,
    PersonalPoisUniversalExecutable,
    PersonalPoisDSSMModel,
    PersonalPoisDSSMOrgs
)
from sandbox.projects.PersonalPoiGenerator.UploadToEcstatic import PoiPackerExecutable, UploadPoiToEcstatic
from sandbox.common import errors

from sandbox.projects.maps.common.juggler_alerts import TaskJugglerReportWithParameters
from sandbox.projects.maps.common.latest_resources import find_latest_resource

from sandbox.projects.maps.poi.BasemapPoiChecker import (
    BasemapPoiCheckerBinary,
    BasemapPoiCheckerYqlUdf,
    BasemapPoiCheckerTask
)


TASK_WAIT_STATUSES = [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK]


class PersonalPoisFeaturesConfigJson(sdk2.Resource):
    any_arch = True


class PersonalPoisConveyor(TaskJugglerReportWithParameters):
    """Conveyor for collect and upload personal pois"""

    class Parameters(TaskJugglerReportWithParameters.Parameters):
        with sdk2.parameters.RadioGroup('Environment') as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

        today = sdk2.parameters.String(
            'Date to calculate the pipeline for',
        )
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            pool = sdk2.parameters.String('YT pool to utilize', default='maps-core-personalized-poi-renderer', required=True)
            yt_vault_token = sdk2.parameters.String('Your yt token name in vault', default='yt-token', required=True)
            yql_vault_token = sdk2.parameters.String('Your yql token name in vault', default='yql-token', required=True)
            yt_common_folder = sdk2.parameters.String(
                'Your folder',
                default='//home/maps/poi/personalized_poi',
                required=True
            )
            yt_data_folder = sdk2.parameters.String(
                'Folder with data tables',
                default='//home/maps/poi/data',
                required=True)

        with sdk2.parameters.Group('Statface parameters') as statface_parameters:
            statface_vault_token = sdk2.parameters.String(
                'Your statface token name in vault',
                default='statface-token', required=True)
            statface_folder = sdk2.parameters.String(
                'Statface folder that should contain all reports',
                default='Maps.Wiki/PersonalizedPOI', required=True)

            with sdk2.parameters.RadioGroup("Statface staging") as statface_staging:
                statface_staging.values.testing = statface_staging.Value(value='testing')
                statface_staging.values.stable = statface_staging.Value(value='stable')

        with sdk2.parameters.Group('Conveyor parameters') as conveyor_parameters:
            universal_task_executable = sdk2.parameters.Resource(
                'Universal executable',
                resource_type=PersonalPoisUniversalExecutable,
                required=False
            )
            features_config = sdk2.parameters.Resource(
                'Json config for prepare feature stage',
                resource_type=PersonalPoisFeaturesConfigJson,
                required=True
            )
            experiment_tags = sdk2.parameters.List('Additional experiment tags', sdk2.parameters.String, default=[])
            experiment_tags_no_zoom = sdk2.parameters.List(
                'Experiment tags which should not calculate zooms manually and use main user pois',
                sdk2.parameters.String,
                default=[],
            )
            external_data_tags = sdk2.parameters.List(
                'External data tags which should be applied to the main experiment',
                sdk2.parameters.String,
                default=[],
            )

        with sdk2.parameters.Group('Trusted users preparator parameters') as trusted_users_preparator_parameters:
            abc_services_table = sdk2.parameters.String(
                'ABC services table',
                default="//home/abc/db/services_servicemember"
            )
            staff_table = sdk2.parameters.String(
                'Staff table',
                default="//home/abc/db/intranet_staff"
            )
            staff_puid_gluing_table = sdk2.parameters.String(
                'Table with crypta gluing staff logins with puid',
                default="//home/crypta/public/ids_storage/staff/puid"
            )
            poi_abc_id = sdk2.parameters.Integer(
                'POI ABC id',
                default=7578
            )
            abc_role_ids = sdk2.parameters.List(
                'ABC role ids',
                sdk2.parameters.Integer,
                default=[4, 8]
            )

        with sdk2.parameters.Group('Basemap POI checker parameters') as basemap_poi_checker_parameters:
            basemap_poi_checker_binary = sdk2.parameters.Resource(
                'Sandbox resource ID of poi checker executable',
                resource_type=BasemapPoiCheckerBinary,
                required=False
            )
            basemap_poi_checker_yql_udf = sdk2.parameters.Resource(
                'Sandbox resource ID of poi checker .so yql_udf',
                resource_type=BasemapPoiCheckerYqlUdf,
                required=False
            )

        with sdk2.parameters.Group('DSSM parameters') as dssm_parameters:
            dssm_model = sdk2.parameters.Resource(
                'DSSM model',
                resource_type=PersonalPoisDSSMModel,
                required=True
            )
            dssm_orgs = sdk2.parameters.Resource(
                'DSSM organizations',
                resource_type=PersonalPoisDSSMOrgs,
                required=True
            )

        do_ecstatic_upload = sdk2.parameters.Bool(
            'Do you want to load tables to ecstatic-storage?',
            default=False,
        )
        with do_ecstatic_upload.value[True]:
            with sdk2.parameters.Group('Ecstatic upload parameters') as ecstatic_parameters:
                ecstatic_envs = sdk2.parameters.List('Ecstatic environments', default=['stable', 'testing'])
                ecstatic_version = sdk2.parameters.Integer('Version-sufix', default=0)

                ecstatic_main_dir = sdk2.parameters.String(
                    'Relative to common-folder dir with org and user-poi tables',
                    default='ecstatic',
                    required=True
                )
                ecstatic_exp_dir = sdk2.parameters.String(
                    'Relative to common-folder dir with experiment poi tables',
                    default='experiment',
                )
                ecstatic_exp_subdir = sdk2.parameters.String(
                    'Subdir for experiment in which to look for poi-tables.',
                    default='ecstatic-history',
                    required=True
                )
                ecstatic_packing_binary = sdk2.parameters.Resource(
                    'Sandbox resource ID for packer binary',
                    resource_type=PoiPackerExecutable,
                    required=False
                )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def _create_subtask(self, task_type, params, description, use_sdk2=False):
        if not use_sdk2:
            task = self.server.task({
                'type': task_type,
                'context': params,
                'children': True,
            })
            self.server.task[task['id']].update({
                'description': description,
                'owner': self.owner,
                'notifications': self.server.task[self.id].read()["notifications"],
                'priority': {
                    "class": self.Parameters.priority.cls,
                    "subclass": self.Parameters.priority.scls,
                },
            })
            self.server.batch.tasks.start.update([task['id']])
            return task['id']
        else:
            task = task_type(self, description=description, **params)
            task.enqueue()
            return task.id

    def check_subtasks(self):
        if not all(
            task.status == ctt.Status.SUCCESS
            for task in self.find(parent_id=self.id)
        ):
            raise TaskError('Child task failed')

    def ensure_latest_resources_used(self):
        self.universal_task_executable = self.Parameters.universal_task_executable
        if not self.universal_task_executable:
            self.universal_task_executable = find_latest_resource(
                PersonalPoisUniversalExecutable, self.Parameters.environment)

        self.ecstatic_packing_binary = self.Parameters.ecstatic_packing_binary
        if not self.ecstatic_packing_binary:
            self.ecstatic_packing_binary = find_latest_resource(
                PoiPackerExecutable, self.Parameters.environment)

        self.basemap_poi_checker_binary = self.Parameters.basemap_poi_checker_binary
        if not self.basemap_poi_checker_binary:
            self.basemap_poi_checker_binary = find_latest_resource(
                BasemapPoiCheckerBinary, self.Parameters.environment)

        self.basemap_poi_checker_yql_udf = self.Parameters.basemap_poi_checker_yql_udf
        if not self.basemap_poi_checker_yql_udf:
            self.basemap_poi_checker_yql_udf = find_latest_resource(
                BasemapPoiCheckerYqlUdf, self.Parameters.environment)

        logging.info(
            'Working in %s environment', self.Parameters.environment)
        logging.info(
            'Using PersonalPoisUniversalExecutable: %s',
            self.universal_task_executable.id)
        logging.info(
            'Using PoiPackerExecutable: %s',
            self.ecstatic_packing_binary.id)
        logging.info(
            'Using BasemapPoiCheckerBinary: %s',
            self.basemap_poi_checker_binary.id)
        logging.info(
            'Using BasemapPoiCheckerYqlUdf: %s',
            self.basemap_poi_checker_yql_udf.id)

    def on_execute(self):
        # Check subtasks before doing a next stage
        # After sdk2.WaitTask(), self.on_execute() is called again skipping
        # self.memoize_stage blocks that have already been completed in
        # previous calls. So self.check_subtasks() will be called after
        # every WAIT_TASK.
        # https://wiki.yandex-team.ru/sandbox/cookbook/#execution-stages
        self.check_subtasks()
        self.ensure_latest_resources_used()

        with self.memoize_stage.save_day:
            self.Context.additional_kwargs = {
                'resource_id': self.universal_task_executable.id
            } if self.universal_task_executable is not None else {}

            if self.Parameters.today is None or len(str(self.Parameters.today)) == 0:
                self.Context.today = datetime.date.today().strftime('%Y-%m-%d')
            else:
                self.Context.today = str(self.Parameters.today)

        PersonalPoisUniversalTaskPartial = functools.partial(
            PersonalPoisUniversalTask,
            self,
            create_sub_task=False,
            notifications=self.Parameters.notifications,
            pool=self.Parameters.pool,
            yt_vault_token=self.Parameters.yt_vault_token,
            yql_vault_token=self.Parameters.yql_vault_token,
            statface_vault_token=self.Parameters.statface_vault_token,
            today=self.Context.today,
        )

        with self.memoize_stage.init_folders:
            logging.info('Init folders')
            task = PersonalPoisUniversalTaskPartial(
                description='Init folders phase of regular process {}'.format(self.id),
                command='init-folders --common-folder {} --data-folder {}'.format(
                    self.Parameters.yt_common_folder,
                    self.Parameters.yt_data_folder
                ),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            raise sdk2.WaitTask(
                [task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.update_dict:
            logging.info('Update dict')
            task = PersonalPoisUniversalTaskPartial(
                description='Update dict phase of regular process {}'.format(self.id),
                command='update-dict --common-folder {} --permalink-oid permalink-oid --yuid-puid yuid-puid'.format(
                    self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            raise sdk2.WaitTask(
                [task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.extract_info_from_maps:
            logging.info('Extract info')
            maps_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from maps phase of regular process {}'.format(self.id),
                command='extract-info maps --common-folder {} --logs-folder {}'.format(
                    self.Parameters.yt_common_folder,
                    '//home/geosearch-prod/geocube/1d',
                ),
                **self.Context.additional_kwargs
            )
            maps_task.enqueue()

            raise sdk2.WaitTask(
                [maps_task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.extract_info:

            bookmarks_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from bookmarks phase of regular process {}'.format(self.id),
                command='extract-info bookmarks --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            bookmarks_task.enqueue()

            orgvisits_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from orgvisits phase of regular process {}'.format(self.id),
                command='extract-info orgvisits --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            orgvisits_task.enqueue()

            analytics_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from analytics phase of regular process {}'.format(self.id),
                command='extract-info analytics --data-folder {}'.format(
                    self.Parameters.yt_data_folder
                ),
                **self.Context.additional_kwargs
            )
            analytics_task.enqueue()

            base_map_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from base map phase of regular process {}'.format(self.id),
                command=' '.join([
                    'extract-info base-map --common-folder {}'.format(self.Parameters.yt_common_folder),
                    '--base-map //home/maps/poi/maps_pois_export/yandex-maps-renderer-compiled:vmap2-cis1_19.10.29-0',
                ]),
                **self.Context.additional_kwargs
            )
            base_map_task.enqueue()

            common_addresses_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from common addresses phase of regular process {}'.format(self.id),
                command='extract-info common-addresses --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            common_addresses_task.enqueue()

            regulargeo_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from regulargeo phase of regular process {}'.format(self.id),
                command='extract-info regulargeo --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            regulargeo_task.enqueue()

            ymapsdf_positions_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from ymapsdf positions phase of regular process {}'.format(self.id),
                command='extract-info ymapsdf-positions --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            ymapsdf_positions_task.enqueue()

            raise sdk2.WaitTask(
                [bookmarks_task.id, orgvisits_task.id, analytics_task.id, base_map_task.id,
                 common_addresses_task.id, regulargeo_task.id, ymapsdf_positions_task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        # The stage depends from ymapsdf positions now, so can't run this one in parallel
        with self.memoize_stage.extract_info_from_altay:
            altay_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from altay phase of regular process {}'.format(self.id),
                command='extract-info altay --common-folder {} --data-folder {}'.format(
                    self.Parameters.yt_common_folder,
                    self.Parameters.yt_data_folder
                ),
                **self.Context.additional_kwargs
            )
            altay_task.enqueue()

            raise sdk2.WaitTask(
                [altay_task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        # get additional info for altay companies in orginfo
        with self.memoize_stage.add_info_for_altay_companies:
            ymapsdf_task = PersonalPoisUniversalTaskPartial(
                description='Extract info from ymapsdf phase of regular process {}'.format(self.id),
                command='extract-info ymapsdf --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            ymapsdf_task.enqueue()

            basemap_poi_checker_subtask_params = {
                'checker_binary': self.basemap_poi_checker_binary.id,
                'checker_yql_udf': self.basemap_poi_checker_yql_udf.id,
                'yt_vault': self.Parameters.yt_vault_token,
                'yql_vault': self.Parameters.yql_vault_token,
                'pool': self.Parameters.pool,
                'permalinks_table': os.path.join(self.Parameters.yt_data_folder, 'orginfo'),
                'output_table': os.path.join(self.Parameters.yt_data_folder, 'org_basemap_reasons'),
            }
            basemap_poi_checker_task_id = self._create_subtask(
                BasemapPoiCheckerTask,
                basemap_poi_checker_subtask_params,
                'Run basemap POI checker phase of regular process',
                use_sdk2=True,
            )

            raise sdk2.WaitTask(
                [ymapsdf_task.id, basemap_poi_checker_task_id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.join_sources_and_calculate_near_organisations:
            logging.info('Join sources and calculate near organisations')
            join_task = PersonalPoisUniversalTaskPartial(
                description='Join sources phase of regular process {}'.format(self.id),
                command='join-sources --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            join_task.enqueue()

            near_task = PersonalPoisUniversalTaskPartial(
                description='Calculate near organisations phase of regular process {}'.format(self.id),
                command='calculate-near-organisations --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            near_task.enqueue()

            raise sdk2.WaitTask(
                [join_task.id, near_task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.prepare_features:
            logging.info('Prepare features')
            json_config = self.Parameters.features_config

            if json_config is None:
                raise errors.TaskError('No json config founded')

            json_config = sdk2.ResourceData(json_config)

            with open(str(json_config.path)) as f:
                features_config_json = json.dumps(json.load(f), separators=(',', ':'))

            task = PersonalPoisUniversalTaskPartial(
                description='Prepare features phase of regular process {}'.format(self.id),
                command='prepare-features --common-folder {} --config-json {}'.format(
                    self.Parameters.yt_common_folder,
                    features_config_json
                ),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            raise sdk2.WaitTask(
                [task.id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.calculate_recommendations:
            logging.info('Calculate recommendations')

            additional_tasks = list()
            if self.Parameters.experiment_tags is not None:
                for experiment_tag in self.Parameters.experiment_tags:
                    if not experiment_tag.startswith('recommendations'):
                        continue
                    if 'dssm' in experiment_tag:
                        additional_kwargs = dict(self.Context.additional_kwargs)
                        if self.Parameters.dssm_model is None or self.Parameters.dssm_orgs is None:
                            errors.TaskError('DSSM model of organizations not found')
                        additional_kwargs['dssm_model_id'] = self.Parameters.dssm_model.id
                        additional_kwargs['dssm_orgs_id'] = self.Parameters.dssm_orgs.id
                    else:
                        additional_kwargs = self.Context.additional_kwargs

                    logging.info('Calculate recommendations for {}'.format(experiment_tag))
                    additional_task = PersonalPoisUniversalTaskPartial(
                        description='Calculate recommendations phase of regular process {} for {}'.format(self.id, experiment_tag),
                        command='calculate-recommendations --common-folder {} --data-folder {} --experiment-tag {}'.format(
                            self.Parameters.yt_common_folder,
                            self.Parameters.yt_data_folder,
                            experiment_tag),
                        **additional_kwargs
                    )
                    additional_tasks.append(additional_task)
                    additional_task.enqueue()

            raise sdk2.WaitTask(
                map(lambda t: t.id, additional_tasks),
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        experiment_tags_no_zoom = set(self.Parameters.experiment_tags_no_zoom)

        with self.memoize_stage.calculate_zoom:
            logging.info('Calculate zoom')
            task = PersonalPoisUniversalTaskPartial(
                description='Calculate zoom phase of regular process {}'.format(self.id),
                command='calculate-zoom --common-folder {}'.format(self.Parameters.yt_common_folder),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            additional_tasks = list()
            if self.Parameters.experiment_tags is not None:
                for experiment_tag in self.Parameters.experiment_tags:
                    if experiment_tag in experiment_tags_no_zoom:
                        logging.info('Skip calculate zoom for {}'.format(experiment_tag))
                        continue
                    logging.info('Calculate zoom for {}'.format(experiment_tag))
                    additional_tasks.append(PersonalPoisUniversalTaskPartial(
                        description='Calculate zoom phase of regular process {} for {}'.format(self.id, experiment_tag),
                        command='calculate-zoom --common-folder {} --experiment-tag {} --input-table {}'.format(
                            self.Parameters.yt_common_folder,
                            experiment_tag,
                            os.path.join(self.Parameters.yt_common_folder, 'group', 'features_schematized')),
                        **self.Context.additional_kwargs
                    ))
                    additional_tasks[-1].enqueue()

            raise sdk2.WaitTask(
                [task.id] + map(lambda t: t.id, additional_tasks),
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.fill_ecstatic_tables:
            logging.info('Fill ecstatic tables')
            external_data_tags_argument = ' --external-data-tags {}'.format(
                ' '.join(self.Parameters.external_data_tags)
            ) if self.Parameters.external_data_tags else ''
            task = PersonalPoisUniversalTaskPartial(
                description='Fill ecstatic tables phase of regular process {}'.format(self.id),
                command=(
                    'fill-ecstatic-tables'
                    ' --common-folder {}'
                    ' --data-folder {}'
                    ' --abc-services-table {}'
                    ' --staff-table {}'
                    ' --staff-puid-gluing-table {}'
                    ' --poi-abc-id {}'
                    ' --abc-role-ids {}'
                    ' --save-history'.format(
                        self.Parameters.yt_common_folder,
                        self.Parameters.yt_data_folder,
                        self.Parameters.abc_services_table,
                        self.Parameters.staff_table,
                        self.Parameters.staff_puid_gluing_table,
                        self.Parameters.poi_abc_id,
                        ' '.join(str(role_id) for role_id in self.Parameters.abc_role_ids),
                    ) + external_data_tags_argument
                ),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            additional_tasks = list()
            if self.Parameters.experiment_tags is not None:
                for experiment_tag in self.Parameters.experiment_tags:
                    argument_for_no_zoom = ' --input-table {}/group/user_pois'.format(
                        self.Parameters.yt_common_folder
                    ) if experiment_tag in experiment_tags_no_zoom else ''
                    logging.info('Fill ecstatic tables for {}'.format(experiment_tag))
                    additional_tasks.append(PersonalPoisUniversalTaskPartial(
                        description='Fill ecstatic tables phase of regular process {} for {}'.format(self.id,
                                                                                                     experiment_tag),
                        command=(
                            'fill-ecstatic-tables'
                            ' --common-folder {}'
                            ' --data-folder {}'
                            ' --experiment-tag {}'
                            ' --abc-services-table {}'
                            ' --staff-table {}'
                            ' --staff-puid-gluing-table {}'
                            ' --poi-abc-id {}'
                            ' --abc-role-ids {}'
                            ' --save-history'.format(
                                self.Parameters.yt_common_folder,
                                self.Parameters.yt_data_folder,
                                experiment_tag,
                                self.Parameters.abc_services_table,
                                self.Parameters.staff_table,
                                self.Parameters.staff_puid_gluing_table,
                                self.Parameters.poi_abc_id,
                                ' '.join(str(role_id) for role_id in self.Parameters.abc_role_ids),
                            ) + argument_for_no_zoom + external_data_tags_argument
                        ),
                        **self.Context.additional_kwargs
                    ))
                    additional_tasks[-1].enqueue()

            raise sdk2.WaitTask(
                [task.id] + map(lambda t: t.id, additional_tasks),
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.ecstatic_upload_good:
            if not self.Parameters.do_ecstatic_upload:
                logging.info('Not uploading to ecstatic')
                return
            logging.info('Start uploading to ecstatic')
            ecstatic_subtask_params = {
                'environments': self.Parameters.ecstatic_envs,
                'version': self.Parameters.ecstatic_version,
                'yt_token': self.Parameters.yt_vault_token,
                'common_folder': self.Parameters.yt_common_folder,
                'main_dir': self.Parameters.ecstatic_main_dir,
                'exp_dir': self.Parameters.ecstatic_exp_dir,
                'exp_subdir': self.Parameters.ecstatic_exp_subdir,
                'exp_tags': self.Parameters.experiment_tags,
                'packing_binary': self.ecstatic_packing_binary.id,
                'date': self.Context.today,
            }
            task_id = self._create_subtask(
                UploadPoiToEcstatic,
                ecstatic_subtask_params,
                'Upload to ecstatic phase of regular process',
                use_sdk2=True,
            )
            raise sdk2.WaitTask(
                [task_id],
                TASK_WAIT_STATUSES,
                wait_all=True
            )

        with self.memoize_stage.monitoring_phase:
            logging.info('Calculate statistics')
            task = PersonalPoisUniversalTaskPartial(
                description='Calculate statistics of regular process {}'.format(self.id),
                command='calculate-statistics --common-folder {} --statface-staging {} --statface-folder {}'.format(
                    self.Parameters.yt_common_folder,
                    self.Parameters.statface_staging,
                    self.Parameters.statface_folder),
                **self.Context.additional_kwargs
            )
            task.enqueue()

            additional_tasks = []
            if self.Parameters.experiment_tags is not None:
                for experiment_tag in self.Parameters.experiment_tags:
                    logging.info('Calculate statistics for {}'.format(experiment_tag))
                    additional_tasks.append(PersonalPoisUniversalTaskPartial(
                        description='Calculate statistics of regular process {} for {}'.format(self.id, experiment_tag),
                        command='calculate-statistics --common-folder {} --experiment-tag {} --statface-staging {} --statface-folder {}'.format(
                            self.Parameters.yt_common_folder,
                            experiment_tag,
                            self.Parameters.statface_staging,
                            self.Parameters.statface_folder),
                        **self.Context.additional_kwargs
                    ))
                    additional_tasks[-1].enqueue()

            raise sdk2.WaitTask(
                [task.id] + map(lambda t: t.id, additional_tasks),
                TASK_WAIT_STATUSES,
                wait_all=True
            )
