import logging
from sandbox import sdk2
import os

from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
from sandbox.common import errors
from sandbox.projects.resource_types import PERSONAL_POI_GENERATOR_EXECUTABLE


class PersonalPoiGeneratePairs(sdk2.Task):
    '''Generate cid-oid features and make table of pairs'''

    class Parameters(sdk2.Task.Parameters):
        resource_id = sdk2.parameters.LastReleasedResource(
            'Resource Id',
            resource_type=PERSONAL_POI_GENERATOR_EXECUTABLE,
            state=(ctr.State.READY,),
            required=True,
        )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String('Your token name in vault', default='yt-token', required=True)

        with sdk2.parameters.Group('Generator run parameters') as grepper_parameters:
            common_folder = sdk2.parameters.String(
                'Folder in YT for common folder',
                default='//home/geosearch/zhshishkin',
                required=True
            )
            date = sdk2.parameters.String(
                'Day for calculating',
                required=True
            )
            is_saas_reduce = sdk2.parameters.Bool(
                'Do you want prepare tables for saas?',
                required=True

            )
            with is_saas_reduce.value[False]:
                do_map = sdk2.parameters.Bool(
                    'First map phase',
                    default=True,
                )
                probability = sdk2.parameters.Float(
                    'Org1 probability',
                    default=0.7,
                    required=True,
                )
            do_fast_pairs = sdk2.parameters.Bool(
                'Fast pairs making',
                default=True,
            )
            do_fast_bad_pairs = sdk2.parameters.Bool(
                'Fast bad pairs making',
                default=True,
            )
            do_fast_zoom_pairs = sdk2.parameters.Bool(
                'Fast zoom pairs making',
                default=True,
            )
            do_fast_card_pairs = sdk2.parameters.Bool(
                'Fast card pairs making',
                default=True,
            )
            do_fast_mnet_pairs = sdk2.parameters.Bool(
                'Fast mnet pairs making',
                default=True,
            )
            do_fast_deep_pairs = sdk2.parameters.Bool(
                'Fast deep pairs making',
                default=True,
            )

            expired_days = sdk2.parameters.Integer(
                'Expired days for predictions',
                default=7,
                required=True,
            )

    class Requirements(sdk2.Task.Requirements):
        pass

    def on_execute(self):

        with self.memoize_stage.get_runner_phase:

            runner = self.Parameters.resource_id

            if runner is None:
                raise errors.TaskError('No executable founded')
            runner = sdk2.ResourceData(runner)

        with self.memoize_stage.run_phase:
            logging.info('Start run')
            with sdk2.helpers.ProcessLog(self, logger='run generation') as pl:
                env = os.environ.copy()
                env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
                run = [
                    str(runner.path),
                    '--common-folder', self.Parameters.common_folder,
                    '--expired-days', str(self.Parameters.expired_days),
                    '--date', self.Parameters.date,
                ]
                if self.Parameters.is_saas_reduce:
                    if self.Parameters.do_fast_pairs:
                        run += ['--saas', 'good']
                    if self.Parameters.do_fast_bad_pairs:
                        run += ['--saas', 'bad']
                    if self.Parameters.do_fast_zoom_pairs:
                        run += ['--saas', 'zoom']
                    if self.Parameters.do_fast_card_pairs:
                        run += ['--saas', 'card']
                    if self.Parameters.do_fast_mnet_pairs:
                        run += ['--saas', 'mnet']
                    if self.Parameters.do_fast_deep_pairs:
                        run += ['--saas', 'deepw']
                        run += ['--saas', 'deepwo']

                else:

                    run += ['--probability', str(self.Parameters.probability)]
                    if self.Parameters.do_map:
                        run.append('--map')
                    if self.Parameters.do_fast_pairs:
                        run.append('--fast-squeeze')
                    if self.Parameters.do_fast_bad_pairs:
                        run.append('--fast-squeeze-with-poi')
                    if self.Parameters.do_fast_zoom_pairs:
                        run.append('--fast-squeeze-zoom-poi')
                    if self.Parameters.do_fast_card_pairs:
                        run.append('--fast-squeeze-card-poi')
                    if self.Parameters.do_fast_mnet_pairs:
                        run.append('--fast-squeeze-mnet-poi')
                    if self.Parameters.do_fast_deep_pairs:
                        run.append('--fast-squeeze-deep-poi')

                ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
                if ret:
                    raise errors.TaskError('Generation failed')
