import logging
from sandbox import sdk2
import os
import json
import datetime

from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
from sandbox.common import errors


class PersonalPoiMetricsExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoiMetricsResult(sdk2.Resource):
    pass


class PersonalPoiMetrics(sdk2.Task):
    '''Calculate and return metrics'''

    class Parameters(sdk2.Task.Parameters):
        resource_id = sdk2.parameters.LastReleasedResource(
            'Resource Id',
            resource_type=PersonalPoiMetricsExecutable,
            state=(ctr.State.READY,),
            required=True,
        )
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String('Your token name in vault', default='yt-token', required=True)
            yt_proxy = sdk2.parameters.String('YT proxy', default='hahn.yt.yandex.net', required=True)
        with sdk2.parameters.Group('Metrics run parameters') as grepper_parameters:
            is_interval = sdk2.parameters.Bool(
                'Do you want choose date interval?',
                required=True
            )
            with is_interval.value[True]:
                first_day = sdk2.parameters.String(
                    'First day of time interval',
                    default='2017-04-01',
                    required=True
                )
                last_day = sdk2.parameters.String(
                    'Last day of time interval',
                    default='2017-04-01',
                    required=True
                )

            percentage = sdk2.parameters.Float(
                'Percentage of logs',
                required=True,
                default=1
            )
            radius = sdk2.parameters.Float(
                'Radius of area',
                required=True
            )
            tags_to_test_ids = sdk2.parameters.Dict(
                'Tag to test ids',
                required=True
            )
            test_ids = sdk2.parameters.List(
                'Test ids',
                default=[]
            )

    class Requirements(sdk2.Task.Requirements):
        pass

    def on_execute(self):

        if self.Parameters.resource_id is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(self.Parameters.resource_id)

        with self.memoize_stage.run_phase:
            logging.info('Start run')

            file_resource_data = sdk2.ResourceData(PersonalPoiMetricsResult(
                self, self.Parameters.description, 'poi-metrics.jsonl',
            ))

            if self.Parameters.is_interval:
                first_day = self.Parameters.first_day
                last_day = self.Parameters.last_day
            else:
                previous_day = (datetime.date.today() - datetime.timedelta(days=1)).strftime('%Y-%m-%d')
                first_day = last_day = previous_day

            run = [
                str(runner.path),
                '--begin', first_day,
                '--end', last_day,
                '--file', str(file_resource_data.path),
                '--dist', str(self.Parameters.radius),
                '--part', str(self.Parameters.percentage),
                '--tags-to-ids',
                json.dumps({key: value.split(',') for key, value in self.Parameters.tags_to_test_ids.items()})
            ]

            if self.Parameters.test_ids:
                run += [
                    '--test-id', ','.join(self.Parameters.test_ids),
                ]

            with sdk2.helpers.ProcessLog(self, logger='run grep') as pl:
                env = os.environ.copy()
                env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)

                ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
                if ret:
                    raise errors.TaskError('Grep failed')

            file_resource_data.ready()
