import os
from sandbox import sdk2
import datetime

import sandbox.sandboxsdk.environments as environments
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common import errors
import sandbox.common.types.client as ctc


class PersonalPoisDumpToYtExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisDumpToJsonlExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisMapStorageReader(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisDumpPoisFromMap(sdk2.Task):
    '''Dump pois on rendered map to yt'''

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=395236579,
            required=True
        )
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String('Your token name in vault', default='yt-token', required=True)
            yt_dumper = sdk2.parameters.LastReleasedResource(
                'YT dumper executable',
                resource_type=PersonalPoisDumpToYtExecutable,
                required=True
            )
            yt_links_folder = sdk2.parameters.String(
                'Path to yt folder for links',
                required=True
            )
            yt_history_folder = sdk2.parameters.String(
                'Path to yt folder for history',
                required=True
            )

        with sdk2.parameters.Group('Json dumper parameters') as dumper_parameters:
            json_dumper = sdk2.parameters.LastReleasedResource(
                'Json dumper executable',
                resource_type=PersonalPoisDumpToJsonlExecutable,
                required=True
            )

            storage_reader = sdk2.parameters.LastReleasedResource(
                'Storage reader',
                resource_type=PersonalPoisMapStorageReader,
                required=True
            )

            map_tag = sdk2.parameters.String(
                'Map tag after yandex-maps-renderer-compiled',
                required=True,
                default='vmap2-cis1'
            )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        client_tags = ctc.Tag.IVA

    def on_execute(self):
        import yt.wrapper as yt

        yt.config['token'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
        yt.config['proxy']['url'] = 'hahn.yt.yandex.net'

        with self.memoize_stage.test_ecstatic_connection:
            with sdk2.helpers.ProcessLog(self, logger='ecstatic connection') as pl:
                ret = sp.Popen(
                    'echo "sandbox" > /var/cache/yandex/maps/hostinfo/canonical_names',
                    shell=True, stdout=pl.stdout, stderr=pl.stderr).wait()
                if ret:
                    raise errors.TaskError('cant add fixing file')

        with self.memoize_stage.get_ecstatic_map:
            with sdk2.helpers.ProcessLog(self, logger='ecstatic log') as pl:
                ret = sp.check_output(
                    "ecstatic -C ecstatic.maps.yandex.net versions --with-status yandex-maps-renderer-compiled stable" +
                    " | grep " + str(self.Parameters.map_tag) + ".*\[A\] | awk '{print $1}'",
                    shell=True, stderr=pl.stderr
                )

                ret = ret.strip()
                if not ret:
                    raise errors.TaskError('run is failed')

                self.Context.map_name = ret
                self.Context.yt_table = os.path.join(self.Parameters.yt_history_folder, ret.replace('=', '_'))

        if not yt.exists(self.Context.yt_table):
            with self.memoize_stage.download_map:
                if self.Context.map_name.replace('=', '_'):
                    with sdk2.helpers.ProcessLog(self, logger='ecstatic log') as pl:
                        ret = sp.Popen(
                            ["ecstatic", '-C', 'ecstatic.maps.yandex.net', "download", self.Context.map_name],
                            stdout=pl.stdout, stderr=pl.stderr).wait()
                        if ret:
                            raise errors.TaskError('run is failed {}'.format(ret))

            with self.memoize_stage.json_dump:
                archive = sp.check_output(
                    "ls {} | grep .tar".format(self.Context.map_name.replace('=', '_')),
                    shell=True, stderr=sp.STDOUT
                )
                archive = archive.strip()
                if not archive:
                    raise errors.TaskError('run is failed')
                json_run = [
                    str(sdk2.ResourceData(self.Parameters.json_dumper).path),
                    str(sdk2.ResourceData(self.Parameters.storage_reader).path),
                    self.Context.map_name.replace('=', '_') + '/' + archive,
                ]
                with open('output.jsonl', 'w') as f, sdk2.helpers.ProcessLog(self, logger='json dump') as pl:
                    ret = sp.Popen(json_run, stdout=f, stderr=pl.stderr).wait()
                    if ret:
                        raise errors.TaskError('run is failed {}'.format(ret))

            with self.memoize_stage.yt_dump:
                env = os.environ.copy()
                env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
                yt_run = [
                    str(sdk2.ResourceData(self.Parameters.yt_dumper).path),
                    '--input-file', 'output.jsonl',
                    '--result-table', self.Context.yt_table
                ]
                with sdk2.helpers.ProcessLog(self, logger='yt dump') as pl:
                    ret = sp.Popen(yt_run, stdout=pl.stdout, stderr=pl.stderr, env=env).wait()
                    if ret:
                        raise errors.TaskError('dump is failed {}'.format(ret))

        with self.memoize_stage.link_phase:
            yt.link(
                self.Context.yt_table,
                os.path.join(self.Parameters.yt_links_folder, datetime.date.today().strftime('%Y-%m-%d')),
                force=True,
            )

            yt.link(
                self.Context.yt_table,
                os.path.join(self.Parameters.yt_links_folder, 'current'),
                force=True,
            )
