import logging
from sandbox import sdk2
import os

from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn
from sandbox.common import errors


class PersonalPoisUniversalExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisDSSMModel(sdk2.Resource):
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisDSSMOrgs(sdk2.Resource):
    releasable = True
    releasers = ['MAPS-GEOQ-RELEASERS']


class PersonalPoisUniversalTask(sdk2.Task):
    '''Universal task for personal pois service'''

    class Parameters(sdk2.Task.Parameters):
        notification_recipients = sdk2.parameters.List(
            'Notification recipients',
            default=['tswr', 'olegvp'])

        notification_transport = sdk2.parameters.String(
            'Notification transport', choices=['email', 'telegram'],
            default='email')

        triggers = sdk2.parameters.List(
            'Trigger words in log', default=['WARN', 'ERROR', 'CRIT'],
            required=True)

        resource_id = sdk2.parameters.LastReleasedResource(
            'Resource Id',
            resource_type=PersonalPoisUniversalExecutable,
            state=(ctr.State.READY,),
            required=True,
        )

        dssm_model_id = sdk2.parameters.LastReleasedResource(
            'DSSM model Id',
            resource_type=PersonalPoisDSSMModel,
            state=(ctr.State.READY,),
            required=False
        )

        dssm_orgs_id = sdk2.parameters.LastReleasedResource(
            'DSSM organizations Id',
            resource_type=PersonalPoisDSSMOrgs,
            state=(ctr.State.READY,),
            required=False
        )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            pool = sdk2.parameters.String('YT pool to utilize', default='maps-core-personalized-poi-renderer', required=True)
            yt_vault_token = sdk2.parameters.String('Your yt token name in vault', default='yt-token', required=True)
            yql_vault_token = sdk2.parameters.String('Your yql token name in vault', default='yql-token', required=True)
        with sdk2.parameters.Group('Statface parameters') as statface_parameters:
            statface_vault_token = sdk2.parameters.String('Your statface token name in vault', default='statface-token', required=True)
        with sdk2.parameters.Group('Binary additional parameters') as cmd:
            today = sdk2.parameters.String(
                'Date (in %Y-%m-%d format)'
            )

            command = sdk2.parameters.String(
                'Parameters',
                required=True
            )

        kill_timeout = 10 * 60 * 60

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def _send_notification(self, text):
        if self.Parameters.notification_transport == 'email':
            self.server.notification(
                subject='PersonalPoiGenerator/UniversalTask notification',
                body=text,
                recipients=self.Parameters.notification_recipients,
                transport=ctn.Transport.EMAIL,
                urgent=True)
        elif self.Parameters.notification_transport == 'telegram':
            self.server.notification(
                body=text,
                recipients=self.Parameters.notification_recipients,
                transport=ctn.Transport.TELEGRAM)

    def _is_notification_line(self, line):
        return any(s in line for s in self.Parameters.triggers)

    def _process_triggers_in_log(self, cmd, path):
        with open(path, 'r') as log:
            notification_lines = filter(self._is_notification_line, log)
            if notification_lines:
                msg = 'Command {} failed:\n{}'.format(
                    cmd, '\n'.join(notification_lines))
                self._send_notification(msg)

    def on_execute(self):
        runner = self.Parameters.resource_id

        if runner is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(runner)

        dssm_model = self.Parameters.dssm_model_id
        dssm_orgs = self.Parameters.dssm_orgs_id
        if dssm_model is not None and dssm_orgs is not None:
            dssm_model = sdk2.ResourceData(dssm_model)
            dssm_orgs = sdk2.ResourceData(dssm_orgs)

        logging.info('Start run')
        with sdk2.helpers.ProcessLog(self, logger='run binary') as pl:
            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            env['YQL_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token)
            env['STATFACE_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.statface_vault_token)
            run = [str(runner.path)]
            if self.Parameters.today:
                run += ['--today', self.Parameters.today]
            if self.Parameters.pool:
                run += ['--pool', self.Parameters.pool]
            run += self.Parameters.command.split()

            if dssm_model is not None and dssm_orgs is not None:
                run.extend(['--dssm-model', str(dssm_model.path),
                            '--dssm-orgs', str(dssm_orgs.path)])

            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            pl.close()
            self._process_triggers_in_log(run, str(pl.stdout.path))
            if ret:
                raise errors.TaskError('run is failed')
