import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.maps.common.ecstatic_bin import MapsEcstaticToolMixin

import subprocess
import shutil

from datetime import datetime
from os import path, environ


DATASET_NAME = 'yandex-maps-perspoi-renderer'
BRANCH = 'stable'
TVM_ID_DATATESTING = 2019479
TVM_SECRET_ID_DATATESTING = 'sec-01e4qzwj5j9ybygg02zc5f41d5'
TVM_ID_STABLE = 2019477
TVM_SECRET_ID_STABLE = 'sec-01e4qzw08mx9y9jf19ndy4n93k'


class PoiPackerExecutable(sdk2.Resource):
    releasable = True
    executable = True
    releasers = ['MAPS-GEOQ-RELEASERS']
    release_subscribers = ['kremius']


class UploadPoiToEcstatic(MapsEcstaticToolMixin, sdk2.Task):
    """Packs organisations and user-poi tables into fbs and upload to ecstatic"""
    class Requirements(sdk2.Task.Requirements):
        ram = 500 << 10  # 500 GB
        disk_space = 300 << 10
        client_tags = ctc.Tag.GENERIC & (ctc.Tag.SAS | ctc.Tag.VLA)

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60 * 60
        owner = 'MAPS'
        environments = sdk2.parameters.List('Ecstatic environments', default=['stable', 'testing'])
        version = sdk2.parameters.Integer('version', default=0)
        yt_token = sdk2.parameters.Vault('yt token vault', required=True)

        common_folder = sdk2.parameters.String(
            'Root folder of POI project',
            default='//home/maps/poi/personalized_poi',
            required=True
        )
        main_dir = sdk2.parameters.String(
            'Relative to common-folder dir with org and user-poi tables',
            default='ecstatic',
            required=True
        )
        exp_dir = sdk2.parameters.String(
            'Relative to common-folder dir with experiment poi tables',
            default='experiment'
        )
        exp_subdir = sdk2.parameters.String(
            'Subdir for experiment in which to look for poi-tables.',
            default='ecstatic-history',
            required=True
        )
        exp_tags = sdk2.parameters.List('Experiments to consider')
        packing_binary = sdk2.parameters.Resource(
            'Sandbox resource ID for packer binary',
            resource_type=PoiPackerExecutable,
            required=True
        )
        date = sdk2.parameters.String(
            'Date for which tables are considered, %Y-%m-%d',
            default=datetime.strftime(datetime.today(), '%Y-%m-%d'),
            required=True
        )

    def on_execute(self):
        resource = sdk2.Resource[self.Parameters.packing_binary]
        resource_data = sdk2.ResourceData(resource)
        shutil.copyfile(str(resource_data.path), "./packing_binary")
        cmd = [
            './packing_binary',
            '--date',
            self.Parameters.date,
            '--main-dir',
            path.join(self.Parameters.common_folder, self.Parameters.main_dir),
            '--exp-dir',
            path.join(self.Parameters.common_folder, self.Parameters.exp_dir),
            '--exp-subdir',
            self.Parameters.exp_subdir,
        ]
        for tag in self.Parameters.exp_tags:
            if tag != '':
                cmd.append('--exp-tags')
                cmd.append(tag)
        environ['YT_TOKEN'] = self.Parameters.yt_token.data()
        subprocess.check_call('chmod +x ./packing_binary', shell=True)
        subprocess.check_call(cmd)
        subprocess.check_call('mkdir data', shell=True)
        subprocess.check_call('mv user_poi.fb data/ && mv organizations.fb data/ && mv trusted_users.fb data/', shell=True)
        version = '{}-{}'.format(
            self.Parameters.date[2:].replace('-', '.'),
            str(self.Parameters.version)
        )
        with open('data/version', 'w') as f:
            f.write(version)
        try:
            for env in self.Parameters.environments:
                args = ['upload', '{}={}'.format(DATASET_NAME, version),
                        'data', '+{}'.format(BRANCH)]
                tvm_id = TVM_ID_DATATESTING if env == 'datatesting' else TVM_ID_STABLE
                tvm_secret_id = TVM_SECRET_ID_DATATESTING if env == 'datatesting' else TVM_SECRET_ID_STABLE

                self.ecstatic(
                    env,
                    args,
                    tvm_id=tvm_id,
                    tvm_secret_id=tvm_secret_id,
                )
        except subprocess.CalledProcessError as e:
            self.set_info(e.output)
            raise TaskFailure('Ecstatic returned ' + str(e.returncode))
