#!/usr/bin/env python

import os

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects import GetRandomRequests as grr
from sandbox.projects.common import utils


GET_RANDOM_QUERIES_BIN_TASK = "get_random_queries_bin_task"
GET_RANDOM_QUERIES_BIN_RES = "get_random_queries_bin_res"
RANDOM_QUERIES_TASK_FMT = "random_queries_{}_{}_task"
RANDOM_QUERIES_RES_FMT = "random_queries_{}_{}_res"

ISO_TO_REGION = {
    "RU": 225,
    "KZ": 159,
    "UA": 187,
    "BY": 149,
    "TR": 983,
    "ALL": 0,
}

REGION_TO_ISO = {v: k for k, v in ISO_TO_REGION.iteritems()}


class BuildTargets(sp.SandboxStringParameter):
    name = "build_targets"
    description = "Targets to build separated by semicolon"
    default_value = ""
    required = False
    group = "Binaries"


class BuildTaskDescriptionFormat(sp.SandboxStringParameter):
    name = "build_task_description_format"
    description = "Format (in terms of Python) to be used to form a description. One argument <target> will be passed."
    default_value = "Build {}"
    required = False
    group = "Binaries"


class QueriesToCollect(sp.SandboxStringParameter):
    name = "queries_to_collect"
    description = "'|'-separated list in format [<REGION>:]<COUNT>. Region can be omitted or ISO code or integer geo_id. Recognized ISO codes: " + ','.join(sorted(ISO_TO_REGION.keys())) + "."
    default_value = ""
    required = False
    group = "Queries"


class MaxQueriesCollectorsAtOnce(sp.SandboxIntegerParameter):
    name = "max_queries_collectors_at_once"
    description = "Maximum number of simultaneously working GET_RANDOM_REQUESTS tasks"
    default_value = 3
    required = False
    group = "Queries"


def do_good_char(c):
    allowed = "abcdefghijklmnopqrstuvwxyz0123456789_"
    c = c.lower()
    return c if c in allowed else '_'


class PrepareQueriesAndBinariesForRearrangeDynamicTests(SandboxTask):
    """ A task to prepare users queries and binaries for RearrangeAcceptance2 tests, TestAllTdiAndRearrange """
    type = "PREPARE_QUERIES_AND_BINARIES_FOR_REARRANGE_DYNAMIC_TESTS"

    input_parameters = [
        BuildTargets,
        BuildTaskDescriptionFormat,
        QueriesToCollect,
        MaxQueriesCollectorsAtOnce,
    ]

    def _launch_builder(self, t):
        key_base = "build_" + "".join([do_good_char(c) for c in t])
        task_key = key_base + "_task"
        res_key = key_base + "_res"

        if task_key in self.ctx:
            return None

        task_id = self.create_subtask(
            task_type="BUILD_ARCADIA_BINARY",
            arch="linux",
            execution_space=20000,  # 20G
            description=self.ctx.get(BuildTaskDescriptionFormat.name, BuildTaskDescriptionFormat.default_value).format(t),
            input_parameters={
                "notify_if_finished": "",
                "build_path": t,
                "binaries": t + "/" + os.path.basename(t),
            }
        ).id
        self.ctx["launched_tasks"].append({
            'task_key': task_key,
            'res_key': res_key,
            'task_id': task_id,
            'res_id': None,
        })
        self.ctx[task_key] = task_id
        return task_id

    def _launch_builders(self):
        task_ids = []
        targets = [t.strip() for t in self.ctx.get(BuildTargets.name, "").split(';') if t.strip()]
        for t in targets:
            task_id = self._launch_builder(t)
            if task_id:
                task_ids.append(task_id)
        return task_ids

    def _launch_queries_binary(self):
        task_key = GET_RANDOM_QUERIES_BIN_TASK

        if task_key in self.ctx:
            return None

        task_id = self.create_subtask(
            task_type="BUILD_ARCADIA_BINARY",
            arch="linux",
            execution_space=20000,  # 20G
            description="Build get_random_queries binary for {} task #{}".format(self.type, self.id),
            input_parameters={
                "notify_if_finished": "",
                "build_path": "yweb/freshness/get_random_queries",
                "binaries": "yweb/freshness/get_random_queries/get_random_queries",
            }
        ).id
        self.ctx["launched_tasks"].append({
            'task_key': task_key,
            'res_key': GET_RANDOM_QUERIES_BIN_RES,
            'task_id': task_id,
            'res_id': None,
        })
        self.ctx[task_key] = task_id
        return task_id

    def _launch_query(self, q):
        if ":" in q:
            geo_id = q.split(":")[0].strip()
            if geo_id in ISO_TO_REGION:
                geo_id = ISO_TO_REGION[geo_id]
            geo_id = int(geo_id)
            queries_number = int(q.split(":")[1].strip())
        else:
            geo_id = 0
            queries_number = int(q.strip())
        geo_iso = REGION_TO_ISO.get(geo_id, str(geo_id))
        task_key = RANDOM_QUERIES_TASK_FMT.format(geo_iso, queries_number)
        res_key = RANDOM_QUERIES_RES_FMT.format(geo_iso, queries_number)

        if task_key in self.ctx:
            return None

        task_id = self.create_subtask(
            task_type="GET_RANDOM_REQUESTS",
            arch="linux",
            execution_space=10000,  # 10G
            description="{} random requests for '{}'".format(queries_number, geo_iso),
            input_parameters={
                "notify_if_finished": "",
                "kill_timeout": 5 * 60 * 60,  # 5 hours
                grr.NumOfRequestParameter.name: queries_number,
                grr.RegionalParameter.name: geo_id,
                grr.ServicesParameter.name: "web",
                grr.LogsTypeParameter.name: "reqans_proto",
                grr.RandomQueriesBinary.name: self.ctx[GET_RANDOM_QUERIES_BIN_TASK],
            }
        ).id
        self.ctx["launched_tasks"].append({
            'task_key': task_key,
            'res_key': res_key,
            'task_id': task_id,
            'res_id': None,
        })
        self.ctx[task_key] = task_id
        return task_id

    def _get_capacity_left(self):
        tasks = utils.get_working_subtasks()
        req_tasks = [t for t in tasks if t.type == 'GET_RANDOM_REQUESTS']
        max_count = self.ctx.get(MaxQueriesCollectorsAtOnce.name, 0)
        if max_count <= 0:
            max_count = 100
        return max_count - len(req_tasks)

    def _launch_queries(self):
        task_ids = []
        queries = [q.strip() for q in self.ctx.get(QueriesToCollect.name, "").split("|") if q.strip()]
        if queries:
            self._launch_queries_binary()
        capacity = self._get_capacity_left()
        for q in queries:
            if capacity <= 0:
                break
            task_id = self._launch_query(q)
            if task_id:
                task_ids.append(task_id)
                capacity -= 1
        if capacity <= 0:
            utils.wait_subtasks_stop(wait_all=False)
        return task_ids

    def _launch_tasks(self):
        task_ids = []
        task_ids.extend(self._launch_builders())
        task_ids.extend(self._launch_queries())
        return task_ids

    def _fill_info_on_task(self, t):
        if not t["res_id"]:
            task = channel.sandbox.get_task(t["task_id"])
            if task and task.is_done():
                if task.type == "BUILD_ARCADIA_BINARY":
                    t["res_id"] = task.ctx.get("out_resource_id")
                elif task.type == "GET_RANDOM_REQUESTS":
                    t["res_id"] = task.ctx.get("random_queries_resource_id")
        if t["res_id"] and t["res_key"] not in self.ctx:
            self.ctx[t["res_key"]] = t["res_id"]

    def _fill_info_on_tasks(self):
        for t in self.ctx["launched_tasks"]:
            self._fill_info_on_task(t)

    def on_execute(self):
        if "launched_tasks" not in self.ctx:
            self.ctx["launched_tasks"] = []
        self._fill_info_on_tasks()
        if not self.ctx.get("all_launched"):
            self._launch_tasks()
        utils.wait_all_subtasks_stop()
        self._fill_info_on_tasks()
