# -*- coding: utf-8 -*-

import json

from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects.common.wizard import current_production
from sandbox.projects.tank import offline

from sandbox.projects.common import dolbilka
from sandbox.projects.common.search.basesearch import task as basesearch_task
from sandbox.projects import VideoAnalyzeBasesearchPerformance as analyze_task

from sandbox.projects.VideoTestBasesearchPerformanceBest import VideoTestBasesearchPerformanceBest
from sandbox.projects.VideoTestBasesearchPerformanceBest import RunOnSSDParameter

from sandbox.projects.GetVideoSearchDatabase import GetVideoSearchDatabase
from sandbox.projects.GetVideoMiddlesearchDatabase import GetVideoMiddlesearchDatabase

from sandbox.projects.common.priemka import database

from sandbox.sandboxsdk import parameters


_BASESEARCH_MEMORY_LIMIT_24GB = "mem_limit_24gb"
_BASESEARCH_MEMORY_LIMIT_40GB = "mem_limit_38gb"

_CGROUPS = {
    "mem_limit_24gb": {
        "memory": {
            "low_limit_in_bytes": "18G",
            "limit_in_bytes": "24G",
            "recharge_on_pgfault": "1",
        }
    },
    "mem_limit_38gb": {
        "memory": {
            "low_limit_in_bytes": "37G",
            "limit_in_bytes": "38G",
            "recharge_on_pgfault": "1",
        }
    },
}


class TestVideoRTYServer(parameters.SandboxBoolParameter):
    name = "test_video_rty_server"
    description = 'Test VideoRTY Server'
    default_value = False


class TestVideoArchive(parameters.SandboxBoolParameter):
    name = "test_video_archive"
    description = 'Need Test Video Archive'
    default_value = False


class CGroupParameter(parameters.SandboxSelectParameter):
    name = 'cgroup_used'
    description = 'Use cgroup for testing'
    required = False
    choices = [
        ('None', ''),
        (_BASESEARCH_MEMORY_LIMIT_24GB, _BASESEARCH_MEMORY_LIMIT_24GB),
        (_BASESEARCH_MEMORY_LIMIT_40GB, _BASESEARCH_MEMORY_LIMIT_40GB),
    ]
    default_value = ''


class ArchiveRamDriveParameter(parameters.SandboxBoolParameter):
    name = 'archive_ramdrive_used'
    description = 'Use ramdrive for snippets'
    default_value = False


class OldBasesearchQueriesCustomParams(parameters.SandboxStringParameter):
    name = 'old_basesearch_queries_custom_params'
    description = 'old index basesearch queries custom params'
    default_value = ''


class NewBasesearchQueriesCustomParams(parameters.SandboxStringParameter):
    name = 'new_basesearch_queries_custom_params'
    description = 'new index basesearch queries custom params'
    default_value = ''


class CustomBasesearchExecutable(parameters.ResourceSelector):
    name = 'custom_basesearch'
    description = 'custom basesearch binary'
    resource_type = [
        resource_types.VIDEOSEARCH_EXECUTABLE,
        resource_types.OTHER_RESOURCE,
    ]


class CustomMiddlePlan(parameters.ResourceSelector):
    name = 'custom_middle_plan'
    description = 'custom middlesearch plan'
    resource_type = [
        resource_types.VIDEO_MIDDLESEARCH_PLAN,
    ]


class CustomMiddleExecutable(parameters.ResourceSelector):
    name = 'custom_middle_executable'
    description = 'custom middlesearch executable'
    resource_type = [
        resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,
    ]


class CustomWarmupTime(parameters.SandboxIntegerParameter):
    name = 'custom_warmup_time'
    description = 'custom warmup time'
    default_value = 120


class GenSearchPlan(parameters.SandboxBoolParameter):
    name = 'gen_search_plan'
    description = 'Generate search plan'
    default_value = True
    group = 'Plan generation parameters'


class GenFactorsPlan(parameters.SandboxBoolParameter):
    name = 'gen_factors_plan'
    description = 'Generate factors plan'
    default_value = True
    group = 'Plan generation parameters'


_PRIEMKA_SOURCE = media_settings.VideoSettings.PRIEMKA_SOURCE


class PriemkaVideoBasesearchDatabase(database.BasePriemkaDatabaseTask):
    """
        Задача приёмки базы видео поиска.
    """

    type = 'PRIEMKA_VIDEO_BASESEARCH_DATABASE'
    input_parameters = database.BasePriemkaDatabaseTask.input_parameters + \
        (TestVideoRTYServer, TestVideoArchive, CGroupParameter, ArchiveRamDriveParameter,
         OldBasesearchQueriesCustomParams, NewBasesearchQueriesCustomParams,
         CustomBasesearchExecutable, CustomMiddlePlan, CustomMiddleExecutable, CustomWarmupTime,
         GenSearchPlan, GenFactorsPlan)
    client_tags = Tag.LINUX_PRECISE

    def _get_tests(self):
        return []

    def _get_analyzes(self):
        search_analyzes = [(None, 'search')] if utils.get_or_default(self.ctx, GenSearchPlan) else []
        factors_analyzes = [(None, 'factors')] if utils.get_or_default(self.ctx, GenFactorsPlan) else []
        snippets_analyzes = [(None, 'snippets')] if utils.get_or_default(self.ctx, TestVideoArchive) else []
        return search_analyzes + factors_analyzes + snippets_analyzes

    def __get_videosearch_or_videorty(self):
        return (
            resource_types.VIDEORTYSERVER_EXECUTABLE
            if self.ctx[TestVideoRTYServer.name] else
            resource_types.VIDEOSEARCH_EXECUTABLE
        )

    ###
    def _get_basesearch_database_tag(self, age):
        return media_settings.VideoSettings.basesearch_tag(
            media_settings.INDEX_MAIN,
            self.__get_region(age),
            media_settings.VideoSettings.get_nanny_token(self)
        )

    def _get_basesearch_database_task(self):
        return GetVideoSearchDatabase.type

    def _get_basesearch_database_resource(self):
        return resource_types.VIDEO_SEARCH_DATABASE

    def _get_basesearch_database_execution_space(self):
        return media_settings.VideoSettings.EXECUTION_SPACE

    def _get_basesearch_executable(self, age):
        custom_basesearch_executable_resource_id = utils.get_or_default(self.ctx, CustomBasesearchExecutable)
        if custom_basesearch_executable_resource_id:
            return custom_basesearch_executable_resource_id

        basesearch = database.BasePriemkaDatabaseTask._get_basesearch_executable(self, age)
        if basesearch:
            return basesearch

        return utils.get_and_check_last_resource_with_attribute(
            self.__get_videosearch_or_videorty(),
            attr_name='released',
            attr_value='stable').id

    def _get_basesearch_config(self, age):
        config = database.BasePriemkaDatabaseTask._get_basesearch_config(self, age)
        if config:
            return config

        return utils.get_and_check_last_resource_with_attribute(resource_types.VIDEO_SEARCH_CONFIG,
                                                                attr_name='released',
                                                                attr_value='stable').id

    def _get_basesearch_performance_task(self):
        return VideoTestBasesearchPerformanceBest.type

    def _get_basesearch_performance_args(self, query_type):
        if query_type == 'snippets' and self.ctx[ArchiveRamDriveParameter.name]:
            return {RunOnSSDParameter.name: True}
        return {}

    def _get_basesearch_analyze_task(self):
        task_type = analyze_task.VideoAnalyzeBasesearchPerformance.type
        basesearch_params = (
            analyze_task.BASESEARCH1_PARAMS,
            analyze_task.BASESEARCH2_PARAMS,
        )
        plan_params = (
            analyze_task.Plan1Parameter,
            analyze_task.Plan2Parameter,
        )
        return task_type, basesearch_params, plan_params

    def _get_basesearch_analyze_args(self, query_type):
        sub_ctx = {
            dolbilka.DolbilkaExecutorMode.name: 'finger',
            dolbilka.DolbilkaMaximumSimultaneousRequests.name: 20,
            # Enable cgroups when all the sandboxes will have it
            # basesearch_task.CgroupParameter.name: search_cgroups.BASESEARCH_MEMORY_LIMIT_24GB,
            offline.WARMUP_TIME_KEY: utils.get_or_default(self.ctx, CustomWarmupTime),
            offline.SHUTDOWN_TIME_KEY: 1,
        }
        if self.ctx[CGroupParameter.name]:
            cgroup_props = json.dumps(_CGROUPS[self.ctx[CGroupParameter.name]])
            sub_ctx.update({
                analyze_task.BASESEARCH1_PARAMS.Cgroup.name: cgroup_props,
                analyze_task.BASESEARCH2_PARAMS.Cgroup.name: cgroup_props,
            })
        if query_type == "snippets":
            sub_ctx.update({
                dolbilka.DolbilkaExecutorRequestsLimit.name: 200000,  # snippetizer has lower rps than search
            })
            if self.ctx[ArchiveRamDriveParameter.name]:
                sub_ctx.update({
                    basesearch_task.RamdriveEnabledParameter.name: True,
                    basesearch_task.RamdriveSizeParameter.name: media_settings.VideoSettings.SNIPPETS_RAMDRIVE_SIZE,
                    basesearch_task.RamdriveFilesParameter.name: " ".join(
                        media_settings.VideoSettings.SNIPPETS_RAMDRIVE_FILES
                    ),
                })
        elif query_type == "search":
            sub_ctx.update({
                dolbilka.DolbilkaExecutorRequestsLimit.name: 500000,
            })
        elif query_type == "factors":
            sub_ctx.update({
                dolbilka.DolbilkaExecutorRequestsLimit.name: 300000,
            })

        return sub_ctx

    def _get_basesearch_models(self):
        models = database.BasePriemkaDatabaseTask._get_basesearch_models(self)
        if models:
            return models

        return current_production.get_current_production_resource_id(
            media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME,
            resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE,
            media_settings.VideoSettings.get_nanny_token(self)
        )

    def _get_middlesearch_models(self):
        models = database.BasePriemkaDatabaseTask._get_middlesearch_models(self)
        if models:
            return models

        return current_production.get_current_production_resource_id(
            media_settings.VideoSettings.SMIDDLE_NANNY_SERVICE_NAME,
            resource_types.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE,
            media_settings.VideoSettings.get_nanny_token(self)
        )

    def _get_stable_middlesearch_database_resource(self):
        return utils.get_and_check_last_resource_with_attribute(resource_types.VIDEO_MMETA_SHARDMAP,
                                                                attr_name='released',
                                                                attr_value='stable')

    def _get_stable_middlesearch_shard(self):
        stable_resource = self._get_stable_middlesearch_database_resource()
        # source format is VIDMMETA_vidmmeta-000-20161026-200027
        shardmap = stable_resource.attributes['shardmap_name']
        _, _, date, time = shardmap.split('.')[0].split('-')
        return 'vidmmeta-000-{date}-{time}'.format(date=date, time=time)

    def _get_middlesearch_database_task(self):
        return GetVideoMiddlesearchDatabase.type

    def _get_middlesearch_database_resource(self):
        return resource_types.VIDEO_MIDDLESEARCH_INDEX

    def _get_middlesearch_executable(self, age):
        custom_middle_executable_id = utils.get_or_default(self.ctx, CustomMiddleExecutable)
        if custom_middle_executable_id:
            return custom_middle_executable_id

        return current_production.get_current_production_resource_id(
            media_settings.VideoSettings.MMETA_NANNY_SERVICE_NAME_MAN,
            resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,
            media_settings.VideoSettings.get_nanny_token(self)
        )

    def _get_middlesearch_config(self, age):
        testenv_attributes = media_settings.VideoSettings.testenv_resource_attributes(
            media_settings.VideoSettings.MMETA_ID, _PRIEMKA_SOURCE[1])
        return utils.get_and_check_last_resource_with_attribute(resource_types.VIDEO_MIDDLESEARCH_CONFIG,
                                                                testenv_attributes[0],
                                                                testenv_attributes[1]).id

    def _get_middlesearch_data(self, age):
        testenv_attributes = media_settings.VideoSettings.testenv_resource_attributes(
            media_settings.VideoSettings.MMETA_ID, _PRIEMKA_SOURCE[1])
        return utils.get_and_check_last_resource_with_attribute(resource_types.VIDEO_MIDDLESEARCH_DATA,
                                                                testenv_attributes[0],
                                                                testenv_attributes[1]).id

    def _get_middlesearch_plan(self):
        custom_middle_plan_id = utils.get_or_default(self.ctx, CustomMiddlePlan)
        if custom_middle_plan_id:
            return custom_middle_plan_id

        testenv_attributes = media_settings.VideoSettings.testenv_middlesearch_queries_attributes(
            media_settings.VideoSettings.MMETA_ID, _PRIEMKA_SOURCE[1], media_settings.INDEX_MAIN)
        return utils.get_and_check_last_resource_with_attribute(resource_types.VIDEO_MIDDLESEARCH_PLAN,
                                                                testenv_attributes[0],
                                                                testenv_attributes[1]).id

    def __get_region(self, age):
        return _PRIEMKA_SOURCE[1] if age == self._OLD_AGE else media_settings.PRIEMKA_REGION

    def _get_basesearch_queries_custom_params(self, age):
        return (
            self.ctx[OldBasesearchQueriesCustomParams.name]
            if age == self._OLD_AGE else
            self.ctx[NewBasesearchQueriesCustomParams.name]
        )

    def _get_gen_search_plan(self):
        return utils.get_or_default(self.ctx, GenSearchPlan)

    def _get_gen_factors_plan(self):
        return utils.get_or_default(self.ctx, GenFactorsPlan)

    def _get_gen_snippets_plan(self):
        return utils.get_or_default(self.ctx, TestVideoArchive)


__Task__ = PriemkaVideoBasesearchDatabase
