# -*- coding: utf-8 -*-
import time
import logging

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters as sp


class Executable(sp.ResourceSelector):
    resource_type = resource_types.OTHER_RESOURCE
    name = "executable_resource_id"
    description = "Executable to crash"
    required = True


class ProgramArguments(sp.SandboxStringParameter):
    name = "program_arguments"
    description = "Program arguments (must be shell-ready, quoted if necessary)"
    required = False
    default_value = ""


class RunCount(sp.SandboxIntegerParameter):
    name = "run_count"
    description = "Number of runs (10 by default)"
    required = False
    default_value = 10


class ProgramCrasher(task.SandboxTask):
    """
        Run program until it crashes.
        Collects coredumps if any
    """
    type = "PROGRAM_CRASHER"

    input_parameters = (
        Executable,
        ProgramArguments,
        RunCount,
    )

    required_ram = 10000    # 10 Gb
    execution_space = 1000  # 1 Gb

    def on_execute(self):
        executable_resource_id = utils.get_or_default(self.ctx, Executable)
        program_arguments = utils.get_or_default(self.ctx, ProgramArguments)
        run_count = int(utils.get_or_default(self.ctx, RunCount))
        executable_path = self.sync_resource(executable_resource_id)
        command = "{} {}".format(executable_path, program_arguments)

        current_run_count = 0

        while current_run_count < run_count:
            current_run_count += 1
            logging.info("Started %s-th crashing pass", current_run_count)
            program_process = process.run_process(
                command,
                shell=True,
                wait=True,
            )
            if program_process.returncode != 0:
                self.set_info("Program crashed at {}th run".format(current_run_count))
                time.sleep(120)  # enough time to save coredump
                break
